package com.elitesland.cbpl.online.data.repo;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.online.constant.OnlineConstant;
import com.elitesland.cbpl.online.data.entity.OnlineStatisticsDO;
import com.elitesland.cbpl.online.data.entity.QOnlineStatisticsDO;
import com.elitesland.cbpl.online.vo.query.StatisticsPagingParamVO;
import com.elitesland.cbpl.tool.db.SqlUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/11/25
 */
@Component
@RequiredArgsConstructor
public class OnlineStatisticsRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QOnlineStatisticsDO statisticsDO = QOnlineStatisticsDO.onlineStatisticsDO;

    public long updateStatus(OnlineStatisticsDO curr) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(statisticsDO.sid.eq(curr.getSid()));
        predicates.add(statisticsDO.uid.eq(curr.getUid()));
        JPAUpdateClause update = jpaQueryFactory.update(statisticsDO)
                .set(statisticsDO.status, curr.getStatus())
                .set(statisticsDO.ip, curr.getIp())
                .set(statisticsDO.country, curr.getCountry())
                .set(statisticsDO.province, curr.getProvince())
                .set(statisticsDO.city, curr.getCity())
                .set(statisticsDO.lastActiveTime, curr.getLastActiveTime())
                .where(ExpressionUtils.allOf(predicates));
        return update.execute();
    }

    public List<OnlineStatisticsDO> queryCurrentOnline() {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(statisticsDO.deleteFlag.eq(0));
        // 当前在线会话
        predicates.add(statisticsDO.status.eq(OnlineConstant.SESSION_STATUS_ONLINE));
        var jpaQuery = jpaQueryFactory.select(statisticsDO)
                .from(statisticsDO);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return jpaQuery.fetch();
    }

    private Predicate pagingWhere(StatisticsPagingParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(statisticsDO.deleteFlag.eq(0));
        // 仅查询在线用户
        predicates.add(statisticsDO.status.eq(OnlineConstant.SESSION_STATUS_ONLINE));
        if (StrUtil.isNotBlank(query.getUid())) {
            predicates.add(statisticsDO.uid.eq(query.getUid()));
        }
        if (StrUtil.isNotBlank(query.getUsername())) {
            predicates.add(statisticsDO.username.eq(query.getUsername()));
        }
        if (StrUtil.isNotBlank(query.getNickname())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getNickname());
            predicates.add(statisticsDO.nickname.like(likeStr));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public long statisticsCountBy(StatisticsPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(statisticsDO.id)
                .from(statisticsDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch().size();
    }

    public List<OnlineStatisticsDO> statisticsPageBy(StatisticsPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(statisticsDO)
                .from(statisticsDO);
        query.setPaging(jpaQuery);
        query.fillOrders(jpaQuery, statisticsDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch();
    }
}
