package com.elitesland.cbpl.online.service;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.online.constant.OnlineConstant;
import com.elitesland.cbpl.online.data.convert.OnlineLogConvert;
import com.elitesland.cbpl.online.data.repo.OnlineLogRepo;
import com.elitesland.cbpl.online.vo.OnlineRegionVO;
import com.elitesland.cbpl.online.vo.OnlineLogVO;
import com.elitesland.cbpl.tool.log.MDCUtil;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import com.elitesland.cbpl.tool.websocket.handler.domain.OnlineUser;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.dromara.dynamictp.core.DtpRegistry;
import org.dromara.dynamictp.core.executor.DtpExecutor;
import org.slf4j.MDC;
import org.springframework.transaction.annotation.Transactional;

import java.util.Map;

/**
 * @author eric.hao
 * @since 2024/11/19
 */
@Slf4j
@RequiredArgsConstructor
public class OnlineServiceImpl implements OnlineService {

    private final OnlineLogRepo onlineLogRepo;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void record(OnlineUser onlineUser) {
        // MDC上下文
        Map<String, String> ctx = MDC.getCopyOfContextMap();
        // 获取租户编码
        String tenantCode = onlineUser.getTenantCode();
        DtpExecutor dtpExecutor = DtpRegistry.getDtpExecutor(OnlineConstant.TP_ONLINE_POOL_NAME);
        dtpExecutor.execute(() -> {
            MDCUtil.setContextMap(ctx);
            // 租户不隔离
            if (StrUtil.isBlank(tenantCode)) {
                this.executor(onlineUser);
            }
            // 按租户隔离
            else {
                TenantSpiUtil.setCurrentTenant(tenantCode);
                this.executor(onlineUser);
                TenantSpiUtil.resetCurrentTenant();
            }
        });
    }

    private void executor(OnlineUser onlineUser) {
        OnlineLogVO logVO = new OnlineLogVO();
        logVO.setSid(onlineUser.getSessionId());
        logVO.setUid(onlineUser.getUid());
        logVO.setUsername(onlineUser.getUsername());
        logVO.setNickname(onlineUser.getNickname());
        // 登记类型：1上线，0下线；
        logVO.setType(onlineUser.getSession().isOpen() ? 1 : 0);
        OnlineRegionVO region = new OnlineRegionVO();
        region.setIp(onlineUser.getIp());
        logVO.setRegion(region);
        logVO.setNow(onlineUser.getNow());
        onlineLogRepo.save(OnlineLogConvert.INSTANCE.saveParamToDO(logVO));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSync() {
        // 1.读取log表数据，更新在线统计表：
        // - 按时间排序，取最新的一条
        // - 如果是『上线』类型，则进行记录
        // 2.读取活跃状态的操作记录
        // - 如果大于上线时间，则进行覆盖
        // 3.判断最后活跃时间，超出配置的离线时长：
        // - 清空归属地值，认为用户已离线
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void snapshot() {
        // 从在线统计表，定期生成快照
    }

    @Override
    public void deletion() {

    }

    @Override
    public void paging() {

    }
}
