package com.elitesland.cbpl.online.service;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.online.constant.OnlineConstant;
import com.elitesland.cbpl.online.data.convert.OnlineLogConvert;
import com.elitesland.cbpl.online.data.convert.OnlineSnapshotConvert;
import com.elitesland.cbpl.online.data.convert.OnlineStatisticsConvert;
import com.elitesland.cbpl.online.data.entity.OnlineLogDO;
import com.elitesland.cbpl.online.data.entity.OnlineSnapshotDO;
import com.elitesland.cbpl.online.data.entity.OnlineStatisticsDO;
import com.elitesland.cbpl.online.data.repo.*;
import com.elitesland.cbpl.online.vo.query.SnapshotParamVO;
import com.elitesland.cbpl.online.vo.query.StatisticsPagingParamVO;
import com.elitesland.cbpl.online.vo.resp.OnlineLogVO;
import com.elitesland.cbpl.online.vo.resp.OnlineSnapshotVO;
import com.elitesland.cbpl.online.vo.resp.OnlineStatisticsVO;
import com.elitesland.cbpl.tool.db.PagingVO;
import com.elitesland.cbpl.tool.log.MDCUtil;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import com.elitesland.cbpl.tool.websocket.handler.domain.OnlineUser;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.dromara.dynamictp.core.DtpRegistry;
import org.dromara.dynamictp.core.executor.DtpExecutor;
import org.slf4j.MDC;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.elitesland.cbpl.online.constant.OnlineConstant.OPERATION_LOGIN;
import static com.elitesland.cbpl.online.constant.OnlineConstant.OPERATION_LOGOUT;
import static com.elitesland.cbpl.tool.websocket.constant.WebSocketConstant.GUEST_TENANT_CODE;

/**
 * @author eric.hao
 * @since 2024/11/19
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class OnlineServiceImpl implements OnlineService {

    private final OnlineLogRepo onlineLogRepo;
    private final OnlineLogRepoProc onlineLogRepoProc;
    private final OnlineStatisticsRepo statisticsRepo;
    private final OnlineStatisticsRepoProc statisticsRepoProc;
    private final OnlineSnapshotRepo snapshotRepo;
    private final OnlineSnapshotRepoProc onlineSnapshotRepoProc;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void record(OnlineUser onlineUser) {
        // MDC上下文
        Map<String, String> ctx = MDC.getCopyOfContextMap();
        // 获取租户编码
        String tenantCode = onlineUser.getTenantCode();
        DtpExecutor dtpExecutor = DtpRegistry.getDtpExecutor(OnlineConstant.TP_ONLINE_POOL_NAME);
        dtpExecutor.execute(() -> {
            MDCUtil.setContextMap(ctx);
            // 租户不隔离
            if (StrUtil.isBlank(tenantCode) || tenantCode.equals(GUEST_TENANT_CODE)) {
                this.executor(onlineUser);
            }
            // 按租户隔离
            else {
                TenantSpiUtil.setCurrentTenant(tenantCode);
                this.executor(onlineUser);
                TenantSpiUtil.resetCurrentTenant();
            }
        });
    }

    private void executor(OnlineUser onlineUser) {
        OnlineLogVO logVO = new OnlineLogVO();
        logVO.setVisitorId(onlineUser.getVisitorId());
        logVO.setSid(onlineUser.getSessionId().toString());
        logVO.setUid(onlineUser.getUid());
        logVO.setUsername(onlineUser.getUsername());
        logVO.setNickname(onlineUser.getNickname());
        // 登记类型：1上线，0下线；
        logVO.setType(onlineUser.getSession().isChannelOpen() ? OPERATION_LOGIN : OPERATION_LOGOUT);
        // 根据IP查询归属地区 TODO
        logVO.setIp(onlineUser.getIp());
        logVO.setNow(onlineUser.getNow());
        onlineLogRepo.save(OnlineLogConvert.INSTANCE.saveParamToDO(logVO));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSync() {
        // 按时间排序，取最新的一条
        List<OnlineLogDO> logs = onlineLogRepoProc.queryCurrentStatus();
        List<OnlineStatisticsDO> statistics = logs.stream()
                .filter(log -> StrUtil.isNotBlank(log.getVisitorId()))
                .map(OnlineStatisticsConvert.INSTANCE::saveParamToDO)
                .collect(Collectors.toList());
        for (var curr : statistics) {
            // 先更新
            if (statisticsRepoProc.updateStatus(curr) == 0) {
                // 再插入
                statisticsRepo.save(curr);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void snapshot(int type) {
        // 从在线统计表，定期生成快照
        var statistics = statisticsRepoProc.queryCurrentOnline();
        var userCount = statistics.stream().map(OnlineStatisticsDO::getUid).distinct().count();
        OnlineSnapshotDO snapshotDO = new OnlineSnapshotDO();
        // 用户数
        snapshotDO.setUserCount(userCount);
        // 会话数
        snapshotDO.setSessionCount(statistics.size());
        // 统计方式
        snapshotDO.setType(type);
        // 生成快照时间
        snapshotDO.setSnapshotTime(LocalDateTime.now());
        snapshotRepo.save(snapshotDO);
    }

    @Override
    public void deletion() {

    }

    @Override
    public PagingVO<OnlineStatisticsVO> paging(StatisticsPagingParamVO query) {
        long count = statisticsRepoProc.statisticsCountBy(query);
        if (count > 0) {
            var list = statisticsRepoProc.statisticsPageBy(query);
            return new PagingVO<>(count, OnlineStatisticsConvert.INSTANCE.doToVO(list));
        }
        return new PagingVO<>();
    }

    @Override
    public List<OnlineSnapshotVO> querySnapshot(SnapshotParamVO query) {
        List<OnlineSnapshotDO> result = onlineSnapshotRepoProc.querySnapshot(query);
        return OnlineSnapshotConvert.INSTANCE.doToVo(result);
    }
}
