package com.elitesland.cbpl.rosefinch.client.queue.consumer;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.rosefinch.client.constant.RosefinchType;
import com.elitesland.cbpl.rosefinch.client.queue.producer.dto.InstanceDTO;
import com.elitesland.cbpl.rosefinch.client.queue.producer.message.TaskMessage;
import com.elitesland.cbpl.rosefinch.client.util.InstanceBotUtil;
import com.elitesland.cbpl.rosefinch.common.constant.InstanceStatus;
import com.elitesland.cbpl.rosefinch.common.queue.ConsumerQueue;
import com.elitesland.cbpl.rosefinch.client.rpc.InstanceRpcService;
import com.elitesland.cbpl.rosefinch.client.spi.InstanceListener;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.util.StringUtils;
import com.elitesland.cbpl.tool.log.MDCUtil;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import com.elitesland.cbpl.unicom.util.UnicomClient;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.time.LocalDateTime;

import static com.elitesland.cbpl.rosefinch.client.constant.ScheduleClientConstant.SCHEDULE_CONSUMER_TAG;
import static com.elitesland.cbpl.tool.core.util.StringUtils.DEFAULT_EXCEED_LENGTH;

/**
 * @author eric.hao
 * @since 2024/09/29
 */
@Slf4j
@RequiredArgsConstructor
public class ConsumerWrapper {

    @Autowired(required = false)
    private InstanceListener instanceListener;
    @Resource
    private ConsumerQueue consumerQueue;
    @Resource
    private InstanceRpcService instanceRpcService;

    /**
     * 消费处理
     */
    public void consumer(TaskMessage payload) {
        // MDC上下文
        MDCUtil.setContextMap(payload.getParentMdc());
        // 获取租户编码
        String tenantCode = payload.getTenantCode();
        // 租户不隔离
        if (StrUtil.isBlank(tenantCode)) {
            this.executor(payload);
        }
        // 按租户隔离
        else {
            TenantSpiUtil.setCurrentTenant(tenantCode);
            this.executor(payload);
            TenantSpiUtil.resetCurrentTenant();
        }
    }

    /**
     * 任务执行过程
     *
     * @param payload 消息报文
     */
    private void executor(TaskMessage payload) {
        InstanceDTO instance = payload.getInstanceDTO();
        LogUtil.info(payload.getTaskCode(), null, "[ROSEFINCH] 任务(" + payload.getTaskCode() + ") 开始执行：", null, null);
        // 1. 实例执行前的回调方法
        try {
            if (instanceListener != null) {
                instanceListener.start(instance);
            }
        } catch (Exception e) {
            logger.error("[ROSEFINCH] 前置任务执行失败：", e);
        }

        // 2. 实例执行
        Throwable throwable = null;
        // 是否被中断
        boolean interrupted = false;
        try {
            // 更新状态：任务运行中
            instance.setInstanceStatus(InstanceStatus.RUNNING.getCode());
            instance.setExecuteTime(LocalDateTime.now());
            instanceRpcService.update(instance);
            // 反射类型任务
            if (RosefinchType.REFLECT.getCode().compareTo(payload.getTaskType()) == 0) {
                UnicomClient.run(() -> consumerQueue.run(payload), SCHEDULE_CONSUMER_TAG);
            } else {
                UnicomClient.run(() -> consumerQueue.run(payload), payload.getTaskCode());
            }

            instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
        } catch (Exception e) {
            throwable = e;
            InstanceBotUtil.push(payload, e);
            // 捕获识别被中断
            if (ExceptionUtils.getCausedBy(e) instanceof InterruptedException) {
                interrupted = true;
            }
            if (!interrupted) {
                instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
                instance.setErrorMessage(StringUtils.exceed(e.getMessage(), DEFAULT_EXCEED_LENGTH));
                LogUtil.error(payload.getTaskCode(), null,
                        "[ROSEFINCH] 执行异常：",
                        null, e, null
                );
            } else {
                logger.error("[ROSEFINCH] 执行异常：", e);
            }
        }

        // 3. 记录实例执行结果
        if (!interrupted) {
            instance.setEndTime(LocalDateTime.now());
            instanceRpcService.update(instance);
        }
        LogUtil.info(payload.getTaskCode(), null, "[ROSEFINCH] 执行结束.", null, null);

        // 4. 实例执行完后的回调方法
        try {
            if (instanceListener != null) {
                instanceListener.whenComplete(instance, throwable);
            }
        } catch (Exception e) {
            LogUtil.error(payload.getTaskCode(), null,
                    "[ROSEFINCH] 后置任务执行异常：",
                    null, e, null
            );
        }
    }
}