package com.elitesland.cbpl.rosefinch.data.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.rosefinch.data.entity.QRosefinchConfigDO;
import com.elitesland.cbpl.rosefinch.data.entity.QRosefinchInstanceDO;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchInstanceDeleteParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchInstancePagingParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchInstanceSaveParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchInstancePagingVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchInstanceRespVO;
import com.elitesland.cbpl.tool.db.SqlUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/08/06
 */
@Component
@RequiredArgsConstructor
public class RosefinchInstanceRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QRosefinchConfigDO rosefinchConfigDO = QRosefinchConfigDO.rosefinchConfigDO;
    private static final QRosefinchInstanceDO rosefinchInstanceDO = QRosefinchInstanceDO.rosefinchInstanceDO;

    private final QBean<RosefinchInstancePagingVO> rosefinchInstancePagingVO = Projections.bean(
            RosefinchInstancePagingVO.class,
            rosefinchInstanceDO.id,
            rosefinchInstanceDO.masId,
            rosefinchInstanceDO.taskCode.coalesce(rosefinchConfigDO.taskCode).as("taskCode"),
            rosefinchInstanceDO.taskName.coalesce(rosefinchConfigDO.taskName).as("taskName"),
            rosefinchInstanceDO.runType,
            rosefinchInstanceDO.instanceStatus,
            rosefinchInstanceDO.startTime,
            rosefinchInstanceDO.executeTime,
            rosefinchInstanceDO.endTime,
            rosefinchInstanceDO.errorMessage,
            rosefinchInstanceDO.traceId,
            rosefinchInstanceDO.customFields,
            rosefinchInstanceDO.tenantId,
            rosefinchInstanceDO.remark,
            rosefinchInstanceDO.createUserId,
            rosefinchInstanceDO.creator,
            rosefinchInstanceDO.createTime,
            rosefinchInstanceDO.modifyUserId,
            rosefinchInstanceDO.updater,
            rosefinchInstanceDO.modifyTime,
            rosefinchInstanceDO.deleteFlag
    );

    private Predicate pagingWhere(RosefinchInstancePagingParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(rosefinchInstanceDO.deleteFlag.eq(0));
        if (ObjectUtil.isNotNull(query.getTaskCodeName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getTaskCodeName());
            predicates.add(rosefinchInstanceDO.taskCode.eq(query.getTaskCodeName())
                    .or(rosefinchInstanceDO.taskName.like(likeStr)));
        }
        if (StrUtil.isNotBlank(query.getInstanceStatus())) {
            predicates.add(rosefinchInstanceDO.instanceStatus.eq(query.getInstanceStatus()));
        }
        if (ObjectUtil.isNotNull(query.getStartTime())) {
            predicates.add(rosefinchInstanceDO.startTime.eq(query.getStartTime()));
        }
        if (ObjectUtil.isNotNull(query.getEndTime())) {
            predicates.add(rosefinchInstanceDO.endTime.eq(query.getEndTime()));
        }
        if (StrUtil.isNotBlank(query.getErrorMessage())) {
            predicates.add(rosefinchInstanceDO.errorMessage.eq(query.getErrorMessage()));
        }
        if (StrUtil.isNotBlank(query.getTraceId())) {
            predicates.add(rosefinchInstanceDO.traceId.eq(query.getTraceId()));
        }
        // tags不为空，则按条件查询
        if (StrUtil.isNotBlank(query.getTags())) {
            predicates.add(rosefinchConfigDO.taskTag.eq(query.getTags()));
        }
        // 为空时，默认仅查询为null的数据
        else {
            predicates.add(rosefinchConfigDO.taskTag.isNull());
        }
        return ExpressionUtils.allOf(predicates);
    }

    public long rosefinchInstanceCountBy(RosefinchInstancePagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(rosefinchInstanceDO.id)
                .from(rosefinchInstanceDO)
                .leftJoin(rosefinchConfigDO).on(rosefinchConfigDO.id.eq(rosefinchInstanceDO.masId));
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch().size();
    }

    public List<RosefinchInstancePagingVO> rosefinchInstancePageBy(RosefinchInstancePagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(rosefinchInstancePagingVO)
                .from(rosefinchInstanceDO)
                .leftJoin(rosefinchConfigDO).on(rosefinchConfigDO.id.eq(rosefinchInstanceDO.masId));
        query.setPaging(jpaQuery);
        query.fillOrders(jpaQuery, rosefinchInstanceDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch();
    }

    private final QBean<RosefinchInstanceRespVO> rosefinchInstanceVO = Projections.bean(
            RosefinchInstanceRespVO.class,
            rosefinchInstanceDO.id,
            rosefinchInstanceDO.masId,
            rosefinchInstanceDO.taskCode.coalesce(rosefinchConfigDO.taskCode).as("taskCode"),
            rosefinchInstanceDO.taskName.coalesce(rosefinchConfigDO.taskName).as("taskName"),
            rosefinchInstanceDO.runType,
            rosefinchInstanceDO.instanceStatus,
            rosefinchInstanceDO.startTime,
            rosefinchInstanceDO.executeTime,
            rosefinchInstanceDO.endTime,
            rosefinchInstanceDO.errorMessage,
            rosefinchInstanceDO.traceId,
            rosefinchInstanceDO.customFields,
            rosefinchInstanceDO.tenantId,
            rosefinchInstanceDO.remark,
            rosefinchInstanceDO.createUserId,
            rosefinchInstanceDO.creator,
            rosefinchInstanceDO.createTime,
            rosefinchInstanceDO.modifyUserId,
            rosefinchInstanceDO.updater,
            rosefinchInstanceDO.modifyTime,
            rosefinchInstanceDO.deleteFlag
    );

    public RosefinchInstanceRespVO findById(Long instanceId) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(rosefinchInstanceDO.deleteFlag.eq(0));
        predicates.add(rosefinchInstanceDO.id.eq(instanceId));
        var jpaQuery = jpaQueryFactory.select(rosefinchInstanceVO)
                .from(rosefinchInstanceDO)
                .leftJoin(rosefinchConfigDO).on(rosefinchConfigDO.id.eq(rosefinchInstanceDO.masId));
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return jpaQuery.fetchOne();
    }

    public long delete(RosefinchInstanceDeleteParamVO paramVO) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(rosefinchInstanceDO.taskCode.eq(paramVO.getTaskCode()));
        predicates.add(rosefinchInstanceDO.instanceStatus.in(paramVO.getInstanceStatus()));
        if (ObjectUtil.isNotNull(paramVO.getBeforeTime())) {
            predicates.add(rosefinchInstanceDO.startTime.lt(paramVO.getBeforeTime()));
        }
        var delete = jpaQueryFactory.delete(rosefinchInstanceDO);
        delete.where(ExpressionUtils.allOf(predicates));
        return delete.execute();
    }

    public long update(RosefinchInstanceSaveParamVO saveParam) {
        JPAUpdateClause update = jpaQueryFactory.update(rosefinchInstanceDO)
                .set(rosefinchInstanceDO.executeTime, saveParam.getExecuteTime())
                .where(rosefinchInstanceDO.id.eq(saveParam.getId()));
        // 日志链路ID
        if (StrUtil.isNotBlank(saveParam.getTraceId())) {
            update.set(rosefinchInstanceDO.traceId, saveParam.getTraceId());
        }
        // 结束时间
        if (ObjectUtil.isNotNull(saveParam.getEndTime())) {
            update.set(rosefinchInstanceDO.endTime, saveParam.getEndTime());
        }
        // 错误日志
        if (StrUtil.isNotBlank(saveParam.getErrorMessage())) {
            update.set(rosefinchInstanceDO.errorMessage, saveParam.getErrorMessage());
        }
        // 实例状态
        if (StrUtil.isNotBlank(saveParam.getInstanceStatus())) {
            update.set(rosefinchInstanceDO.instanceStatus, saveParam.getInstanceStatus());
        }
        // 更新备注
        if (StrUtil.isNotBlank(saveParam.getRemark())) {
            update.set(rosefinchInstanceDO.remark, saveParam.getRemark());
        }
        return update.execute();
    }
}
