package com.elitesland.cbpl.rosefinch.data.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.rosefinch.data.convert.RosefinchConfigConvert;
import com.elitesland.cbpl.rosefinch.data.entity.RosefinchConfigDO;
import com.elitesland.cbpl.rosefinch.data.repo.RosefinchConfigRepo;
import com.elitesland.cbpl.rosefinch.data.repo.RosefinchConfigRepoProc;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchConfigService;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchConfigPagingParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchConfigSaveParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchConfigDetailVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchConfigPagingVO;
import com.elitesland.cbpl.rosefinch.domain.TaskRequestParam;
import com.elitesland.cbpl.rosefinch.proxy.QueueProxy;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

/**
 * @author eric.hao
 * @since 2024/08/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class RosefinchConfigServiceImpl implements RosefinchConfigService {

    private final RosefinchConfigRepo rosefinchConfigRepo;
    private final RosefinchConfigRepoProc rosefinchConfigRepoProc;
    @Autowired(required = false)
    private QueueProxy queueProxy;

    @Override
    public PagingVO<RosefinchConfigPagingVO> rosefinchConfigPageBy(RosefinchConfigPagingParamVO query) {
        long count = rosefinchConfigRepoProc.rosefinchConfigCountBy(query);
        if (count > 0) {
            var list = rosefinchConfigRepoProc.rosefinchConfigPageBy(query);
            return new PagingVO<>(count, RosefinchConfigConvert.INSTANCE.doToPageVO(list));
        }
        return new PagingVO<>();
    }

    @Override
    public RosefinchConfigDetailVO rosefinchConfigById(Long id) {
        Optional<RosefinchConfigDO> rosefinchConfigDO = rosefinchConfigRepo.findById(id);
        if (rosefinchConfigDO.isEmpty()) {
            throw PhoenixException.unexpected("Not Found Data");
        }
        return RosefinchConfigConvert.INSTANCE.doToVO(rosefinchConfigDO.get());
    }

    @Override
    public TaskRequestParam rosefinchConfigByCode(String taskCode) {
        Optional<RosefinchConfigDO> rosefinchConfigDO = rosefinchConfigRepo.findByTaskCode(taskCode);
        if (rosefinchConfigDO.isEmpty()) {
            throw PhoenixException.unexpected("Not Found Data");
        }
        return RosefinchConfigConvert.INSTANCE.doToRequest(rosefinchConfigDO.get());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(RosefinchConfigSaveParamVO saveParam) {
        boolean exists = rosefinchConfigRepoProc.existsByCode(saveParam.getId(), saveParam.getTaskCode());
        Assert.isFalse(exists, "任务编码已存在");
        // 新增
        if (saveParam.isNew()) {
            RosefinchConfigDO rosefinchConfigDO = RosefinchConfigConvert.INSTANCE.saveParamToDO(saveParam);
            rosefinchConfigRepo.save(rosefinchConfigDO);
            return rosefinchConfigDO.getId();
        }
        // 修改
        else {
            Optional<RosefinchConfigDO> rosefinchConfigDO = rosefinchConfigRepo.findById(saveParam.getId());
            if (rosefinchConfigDO.isEmpty()) {
                throw PhoenixException.unexpected("Not Found Data");
            }
            RosefinchConfigDO rosefinchConfig = rosefinchConfigDO.get();
            RosefinchConfigConvert.INSTANCE.saveParamMergeToDO(saveParam, rosefinchConfig);
            rosefinchConfigRepo.save(rosefinchConfig);
            return rosefinchConfig.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void active(Long id) {
        rosefinchConfigRepoProc.updateStatus(id, ActiveStatus.ACTIVE.getCode());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void inactive(Long id) {
        rosefinchConfigRepoProc.updateStatus(id, ActiveStatus.INACTIVE.getCode());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<Long> ids) {
        rosefinchConfigRepo.deleteAllById(ids);
    }

    @Override
    public void execute(String taskCode) {
        if (ObjectUtil.isNull(queueProxy)) {
            throw new RuntimeException("Rosefinch模块未开启");
        }
        try {
            queueProxy.runImmediately(rosefinchConfigByCode(taskCode));
        } catch (RuntimeException e) {
            throw PhoenixException.unexpected("任务执行失败：" + e.getMessage());
        } catch (Exception e) {
            logger.error("[ROSEFINCH] 任务执行异常: {}", ExceptionUtils.formatException(e));
            throw PhoenixException.unexpected("任务执行异常");
        }
    }
}
