package com.elitesland.cbpl.rosefinch.data.controller;

import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.rosefinch.enums.InstanceStatus;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchConfigService;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchInstanceService;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchInstanceDeleteParamVO;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * @author eric.hao
 * @since 2024/08/07
 */
@Slf4j
@Api(tags = "任务运维操作")
@RestController
@RequestMapping("/rosefinch/devops")
@AllArgsConstructor
public class RosefinchOpsController {

    private final RosefinchConfigService configService;
    private final RosefinchInstanceService instanceService;

    @ApiOperation(value = "删除实例日志")
    @DeleteMapping(value = "/task/delete/instance")
    public HttpResult<Long> delete(@Valid @RequestBody RosefinchInstanceDeleteParamVO paramVO) {
        logger.info("[Rosefinch] delete instance param = {}", paramVO);
        return HttpResult.ok(instanceService.deleteInstance(paramVO, "INVOKE"));
    }

    @ApiOperation(value = "立即执行一次")
    @PostMapping(value = "/task/execute")
    public HttpResult<Void> execute(String taskCode) {
        logger.info("[Rosefinch] execute immediately, taskCode = {}", taskCode);
        configService.execute(taskCode);
        return HttpResult.ok();
    }

    @ApiOperation(value = "立即终止")
    @PostMapping(value = "/instance/stop")
    public HttpResult<Boolean> stop(Long instanceId) {
        logger.info("[Rosefinch] stop immediately, instanceId = {}", instanceId);
        var instance = instanceService.rosefinchInstanceById(instanceId);
        if (ObjectUtil.isNull(instance)) {
            throw new RuntimeException("实例不存在");
        }
        List<String> status = ListUtil.of(InstanceStatus.READY.getCode(), InstanceStatus.RUNNING.getCode());
        if (!status.contains(instance.getInstanceStatus())) {
            throw new RuntimeException(instance.getInstanceStatusName() + "的任务不能终止");
        }
        return HttpResult.ok(configService.stopImmediately(instanceId));
    }
}
