package com.elitesland.cbpl.rosefinch.proxy;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.rosefinch.constant.InstanceStatus;
import com.elitesland.cbpl.rosefinch.constant.SuccessTag;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchInstanceService;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchInstanceSaveParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchInstanceDetailVO;
import com.elitesland.cbpl.rosefinch.domain.TaskRequestParam;
import com.elitesland.cbpl.rosefinch.queue.QueueHandlerService;
import com.elitesland.cbpl.rosefinch.spi.RosefinchListener;
import com.elitesland.cbpl.rosefinch.spi.convert.InstanceDtoConvert;
import com.elitesland.cbpl.rosefinch.util.RosefinchTraceUtil;
import com.elitesland.cbpl.rosefinch.util.RosefinchUtil;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.ReflectionUtils;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.time.LocalDateTime;

import static com.elitesland.cbpl.rosefinch.util.RosefinchUtil.instanceKey;
import static com.elitesland.cbpl.tool.core.util.StringUtils.DEFAULT_EXCEED_LENGTH;
import static com.elitesland.cbpl.tool.tenant.TenantSpiUtil.*;

/**
 * @author eric.hao
 * @since 2024/08/05
 */
@Slf4j
@RequiredArgsConstructor
public class QueueProxy {

    @Autowired(required = false)
    private RosefinchListener rosefinchListener;
    @Resource
    private RosefinchInstanceService instanceService;
    private final QueueHandlerService handlerService;

    /**
     * 实例日志初始化
     *
     * @param request 任务调度参数
     * @param runType 执行方式
     * @return 任务实例
     */
    private RosefinchInstanceSaveParamVO instanceLog(TaskRequestParam request, String runType) {
        var saveParam = new RosefinchInstanceSaveParamVO();
        saveParam.setMasId(request.getTaskId());
        saveParam.setTraceId(RosefinchTraceUtil.getTraceId());
        saveParam.setTaskName(request.getTaskName());
        saveParam.setTaskCode(request.getTaskCode());
        saveParam.setStartTime(LocalDateTime.now());
        saveParam.setInstanceStatus(InstanceStatus.READY.getCode());
        saveParam.setRunType(runType);
        saveParam.setMethodArgs(request.getMethodArgs());
        Long instanceId = instanceService.save(saveParam);
        saveParam.setId(instanceId);
        return saveParam;
    }


    /**
     * 执行任务封装工具类 - CRON
     * <li>记录实例执行情况</li>
     *
     * @param request 任务调度参数
     * @return 封装方法
     */
    public Runnable runAuto(TaskRequestParam request) {
        Runnable executor = () -> runImmediately(request, "自动执行");
        // 租户不隔离
        if (RosefinchUtil.noTenant()) {
            return executor;
        }
        // 按租户隔离
        return () -> byTenantDirectly(executor, currentTenantCode());
    }

    /**
     * 执行任务封装工具类 - 立即执行
     * <li>记录实例执行情况</li>
     *
     * @param request 任务调度参数
     * @return 任务实例ID
     */
    public Long runImmediately(TaskRequestParam request) {
        return runImmediately(request, "手动执行");
    }

    /**
     * 任务执行调度
     *
     * @param request 任务调度参数
     * @param runType 执行方式
     * @return 任务实例ID
     */
    public Long runImmediately(TaskRequestParam request, String runType) {
        RosefinchTraceUtil.initTraceId();
        var instance = instanceLog(request, runType);
        handlerService.publish(instance, request);
        RosefinchTraceUtil.clearTraceId();
        return instance.getId();
    }

    /**
     * 立即终止实例
     *
     * @param instance 实例
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    public boolean stopImmediately(RosefinchInstanceDetailVO instance) {
        try {
            LogUtil.info(instance.getTaskCode(), null, "[ROSEFINCH] 终止任务...", null, null);
            // 立即终止
            boolean stopTag = handlerService.stopImmediately(instanceKey(instance.getId()));
            // 更新执行结果
            var saveParam = new RosefinchInstanceSaveParamVO();
            saveParam.setId(instance.getId());
            saveParam.setEndTime(LocalDateTime.now());
            if (stopTag) {
                saveParam.setInstanceStatus(InstanceStatus.STOP.getCode());
            }
            saveParam.setRemark("手动终止：" + SuccessTag.fromCode(stopTag));
            instanceService.update(saveParam);
            LogUtil.info(instance.getTaskCode(), null, "[ROSEFINCH] 终止任务：" + SuccessTag.fromCode(stopTag), null, null);
            return stopTag;
        } catch (Exception e) {
            logger.error("[ROSEFINCH] 终止任务异常：", e);
            LogUtil.error("[ROSEFINCH] 终止任务异常：", e);
            // 终止异常，按失败返回
            return false;
        }
    }

    /**
     * 任务执行过程
     *
     * @param instance 实例日志
     * @param request  任务调度参数
     */
    public void execute(RosefinchInstanceSaveParamVO instance, TaskRequestParam request) {
        LogUtil.info(instance.getTaskCode(), null, "[ROSEFINCH] 任务(" + instance.getTaskCode() + ") 开始执行：", null, null);
        // 1. 实例执行前的回调方法
        try {
            if (rosefinchListener != null) {
                var instanceDTO = InstanceDtoConvert.INSTANCE.saveParamToVO(instance);
                rosefinchListener.start(instanceDTO);
            }
        } catch (Exception e) {
            logger.error("[ROSEFINCH] 前置任务执行失败：", e);
        }
        // 2. 实例执行
        Throwable throwable = null;
        try {
            // 更新状态：任务运行中
            instance.setInstanceStatus(InstanceStatus.RUNNING.getCode());
            instance.setExecuteTime(LocalDateTime.now());
            instanceService.update(instance);
            // 执行任务
            this.run(request);
            instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
        } catch (Exception e) {
            throwable = e;
            instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
            instance.setErrorMessage(ExceptionUtils.formatException(e, DEFAULT_EXCEED_LENGTH));
            LogUtil.error(instance.getTaskCode(), null,
                    "[ROSEFINCH] 执行异常：",
                    null, e, null
            );
        }

        // 3. 记录实例执行结果
        instance.setEndTime(LocalDateTime.now());
        instanceService.update(instance);
        LogUtil.info(instance.getTaskCode(), null, "[ROSEFINCH] 执行结束.", null, null);

        // 4. 实例执行完后的回调方法
        try {
            if (rosefinchListener != null) {
                var instanceDTO = InstanceDtoConvert.INSTANCE.saveParamToVO(instance);
                rosefinchListener.whenComplete(instanceDTO, throwable);
            }
        } catch (Exception e) {
            LogUtil.error(instance.getTaskCode(), null,
                    "[ROSEFINCH] 后置任务执行异常：",
                    null, e, null
            );
        }
    }

    private void run(TaskRequestParam request) {
        Object bean = SpringUtils.getClassBean(request.getClassName());
        // 无参函数
        if (StrUtil.isNullOrUndefined(request.getMethodArgs())) {
            Method method = ReflectionUtils.findMethod(bean.getClass(), request.getMethod());
            ReflectionUtils.invokeMethod(method, bean);
        }
        // 有参函数，且仅为`String.class`
        else {
            Method method = ReflectionUtils.findMethod(bean.getClass(), request.getMethod(), String.class);
            ReflectionUtils.invokeMethod(method, bean, request.getMethodArgs());
        }
    }
}
