package com.elitesland.cbpl.rosefinch.proxy;

import cn.hutool.core.net.NetUtil;
import com.elitesland.cbpl.rosefinch.client.constant.RosefinchType;
import com.elitesland.cbpl.rosefinch.client.queue.producer.ProducerProxy;
import com.elitesland.cbpl.rosefinch.client.queue.producer.message.TaskMessage;
import com.elitesland.cbpl.rosefinch.common.constant.InstanceStatus;
import com.elitesland.cbpl.rosefinch.constant.RosefinchConstant;
import com.elitesland.cbpl.rosefinch.data.convert.RosefinchConfigConvert;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchInstanceService;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchInstanceSaveParamVO;
import com.elitesland.cbpl.rosefinch.proxy.domain.RosefinchMessage;
import com.elitesland.cbpl.rosefinch.queue.QueueHandlerService;
import com.elitesland.cbpl.rosefinch.util.RosefinchTraceUtil;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.redis.util.RedisUtil;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import com.elitesland.cbpl.unicom.domain.InvokeTag;
import com.lzhpo.tracer.util.TracerUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * @author eric.hao
 * @since 2024/10/08
 */
@Slf4j
@Service
@UnicomTag(InvokeTag.TAG_GENERAL)
@RequiredArgsConstructor
public class ProducerProxyService implements ProducerProxy {

    private final RosefinchInstanceService instanceService;
    private final QueueHandlerService handlerService;
    private final RedisUtil redisUtil;

    @Override
    public Long send(TaskMessage payload, String runType) {
        // 初始化任务链路ID
        RosefinchTraceUtil.initTraceId();
        // 初始化任务日志
        var instance = instanceLog(payload, runType);
        payload.setInstanceDTO(RosefinchConfigConvert.INSTANCE.saveParamToDTO(instance));
        // 生产者：发布报文消息
        handlerService.publish(payload);
        // 销毁任务链路ID
        RosefinchTraceUtil.clearTraceId();
        return instance.getId();
    }

    @Override
    public boolean stop(String taskCode, long instanceId) {
        redisUtil.publishMessage(RosefinchConstant.InstanceChannel.INSTANCE_STOP, this.of(taskCode, instanceId));
        return true;
    }

    /**
     * 实例日志初始化
     *
     * @param payload 消息报文
     * @param runType 执行方式
     * @return 任务实例
     */
    private RosefinchInstanceSaveParamVO instanceLog(TaskMessage payload, String runType) {
        var saveParam = new RosefinchInstanceSaveParamVO();
        saveParam.setMasId(payload.getTaskId());
        saveParam.setTaskName(payload.getTaskName());
        saveParam.setTaskCode(payload.getTaskCode());
        saveParam.setRunType(runType);
        saveParam.setInstanceStatus(InstanceStatus.READY.getCode());
        saveParam.setStartTime(LocalDateTime.now());
        saveParam.setGlobalTraceId(TracerUtils.getTraceId());
        saveParam.setTraceId(RosefinchTraceUtil.getTraceId());
        saveParam.setServerIp(NetUtil.getLocalhostStr());
        // 批处理任务
        if (RosefinchType.BATCH.getCode().compareTo(payload.getTaskType()) == 0) {
            saveParam.setTotalCount(payload.size());
            saveParam.setSuccessCount(0);
            saveParam.setFailCount(0);
        }
        // 其他
        else {
            saveParam.setMethodArgs(BeanUtils.toJsonOrEmpty(payload.getPayload()));
        }
        Long instanceId = instanceService.save(saveParam);
        saveParam.setId(instanceId);
        return saveParam;
    }

    private RosefinchMessage of(String taskCode, long instanceId) {
        RosefinchMessage message = new RosefinchMessage();
        message.setTaskCode(taskCode);
        message.setInstanceId(instanceId);
        message.setTenantCode(TenantSpiUtil.currentTenantCode());
        return message;
    }
}
