package com.elitesland.cbpl.rosefinch.data.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.rosefinch.client.queue.producer.ProducerProxy;
import com.elitesland.cbpl.rosefinch.client.queue.producer.message.TaskMessage;
import com.elitesland.cbpl.rosefinch.data.convert.RosefinchConfigConvert;
import com.elitesland.cbpl.rosefinch.data.entity.RosefinchConfigDO;
import com.elitesland.cbpl.rosefinch.data.repo.RosefinchConfigRepo;
import com.elitesland.cbpl.rosefinch.data.repo.RosefinchConfigRepoProc;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchConfigService;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchConfigPagingParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchConfigQueryParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchConfigSaveParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchConfigDetailVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchConfigPagingVO;
import com.elitesland.cbpl.rosefinch.data.vo.resp.RosefinchInstanceDetailVO;
import com.elitesland.cbpl.rosefinch.spi.RosefinchStatusListener;
import com.elitesland.cbpl.rosefinch.util.RosefinchUtil;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.elitesland.cbpl.tool.tenant.TenantSpiUtil.byTenantDirectly;
import static com.elitesland.cbpl.tool.tenant.TenantSpiUtil.currentTenantCode;

/**
 * @author eric.hao
 * @since 2024/08/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class RosefinchConfigServiceImpl implements RosefinchConfigService {

    private final RosefinchConfigRepo rosefinchConfigRepo;
    private final RosefinchConfigRepoProc rosefinchConfigRepoProc;
    @Autowired(required = false)
    private ProducerProxy producerProxy;
    /**
     * 自定义状态切换方法
     */
    private final List<RosefinchStatusListener> rosefinchStatusListeners;

    @Override
    public PagingVO<RosefinchConfigPagingVO> rosefinchConfigPageBy(RosefinchConfigPagingParamVO query) {
        long count = rosefinchConfigRepoProc.rosefinchConfigCountBy(query);
        if (count > 0) {
            var list = rosefinchConfigRepoProc.rosefinchConfigPageBy(query);
            return new PagingVO<>(count, RosefinchConfigConvert.INSTANCE.doToPageVO(list));
        }
        return new PagingVO<>();
    }

    @Override
    public List<RosefinchConfigDetailVO> rosefinchConfigByParam(RosefinchConfigQueryParamVO query) {
        return rosefinchConfigRepoProc.rosefinchConfigByParam(query);
    }

    @Override
    public RosefinchConfigDetailVO rosefinchConfigById(Long id) {
        Optional<RosefinchConfigDO> rosefinchConfigDO = rosefinchConfigRepo.findById(id);
        if (rosefinchConfigDO.isEmpty()) {
            throw new RuntimeException("Not Found Data");
        }
        return RosefinchConfigConvert.INSTANCE.doToVO(rosefinchConfigDO.get());
    }

    @Override
    public RosefinchConfigDetailVO rosefinchConfigByCode(String taskCode) {
        Optional<RosefinchConfigDO> rosefinchConfigDO = rosefinchConfigRepo.findByTaskCode(taskCode);
        if (rosefinchConfigDO.isEmpty()) {
            throw new RuntimeException("Not Found Data");
        }
        return RosefinchConfigConvert.INSTANCE.doToVO(rosefinchConfigDO.get());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(RosefinchConfigSaveParamVO saveParam) {
        boolean exists = rosefinchConfigRepoProc.existsByCode(saveParam.getId(), saveParam.getTaskCode());
        Assert.isFalse(exists, "任务编码(" + saveParam.getTaskCode() + ")已存在");
        // 新增
        if (saveParam.isNew()) {
            RosefinchConfigDO rosefinchConfigDO = RosefinchConfigConvert.INSTANCE.saveParamToDO(saveParam);
            rosefinchConfigRepo.save(rosefinchConfigDO);
            // 执行自定义状态切换逻辑
            updateStatus(RosefinchConfigConvert.INSTANCE.doToVO(rosefinchConfigDO), ActiveStatus.ACTIVE.getCode());
            return rosefinchConfigDO.getId();
        }
        // 修改
        else {
            Optional<RosefinchConfigDO> rosefinchConfigDO = rosefinchConfigRepo.findById(saveParam.getId());
            if (rosefinchConfigDO.isEmpty()) {
                throw new RuntimeException("Not Found Data");
            }
            RosefinchConfigDO rosefinchConfig = rosefinchConfigDO.get();
            RosefinchConfigConvert.INSTANCE.saveParamMergeToDO(saveParam, rosefinchConfig);
            rosefinchConfigRepo.save(rosefinchConfig);
            // 执行自定义状态切换逻辑
            updateStatus(RosefinchConfigConvert.INSTANCE.doToVO(rosefinchConfig), ActiveStatus.INACTIVE.getCode());
            return rosefinchConfig.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void active(Long id) {
        var rosefinchCfg = rosefinchConfigById(id);
        rosefinchConfigRepoProc.updateStatus(id, ActiveStatus.ACTIVE.getCode());
        updateStatus(rosefinchCfg, ActiveStatus.ACTIVE.getCode());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void inactive(Long id) {
        var rosefinchCfg = rosefinchConfigById(id);
        rosefinchConfigRepoProc.updateStatus(id, ActiveStatus.INACTIVE.getCode());
        updateStatus(rosefinchCfg, ActiveStatus.INACTIVE.getCode());
    }

    private void updateStatus(RosefinchConfigDetailVO rosefinchCfg, Integer status) {
        if (CollUtil.isNotEmpty(rosefinchStatusListeners)) {
            rosefinchStatusListeners.stream().filter(listener -> listener.supports(rosefinchCfg))
                    .forEach(listener -> {
                        // 启用
                        if (ObjectUtil.equals(ActiveStatus.ACTIVE.getCode(), status)) {
                            listener.active(rosefinchCfg);
                        }
                        // 禁用
                        if (Objects.equals(ActiveStatus.INACTIVE.getCode(), status)) {
                            listener.inactive(rosefinchCfg);
                        }
                    });
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<Long> ids) {
        rosefinchConfigRepo.deleteAllById(ids);
    }

    @Override
    public void execute(String taskCode, Object payload) {
        var rosefinchConfigDO = rosefinchConfigRepo.findByTaskCode(taskCode);
        if (rosefinchConfigDO.isEmpty()) {
            throw new RuntimeException("Not Found Data");
        }
        var taskInfo = RosefinchConfigConvert.INSTANCE.doToRequest(rosefinchConfigDO.get());
        execute(TaskMessage.withPayload(taskInfo, payload));
    }

    @Override
    public void execute(TaskMessage payload) {
        if (producerProxy == null) {
            throw new RuntimeException("Rosefinch模块未开启");
        }
        try {
            producerProxy.send(payload);
        } catch (RuntimeException e) {
            throw new RuntimeException("任务执行失败：" + e.getMessage());
        } catch (Exception e) {
            logger.error("[ROSEFINCH] 任务执行异常: ", e);
            throw new RuntimeException("任务执行异常");
        }
    }

    @Override
    public Runnable runAuto(TaskMessage payload) {
        String tenantCode = currentTenantCode();
        if (producerProxy == null) {
            throw new RuntimeException("Rosefinch模块未开启");
        }
        try {
            Runnable executor = () -> {
                try {
                    producerProxy.send(payload, "自动执行");
                    // 任务发布过程比较简单，执行过快会导致分布式锁逻辑失效
                    // 比如多节点定时任务，会重复发布执行相同的任务：成功发布任务后，增加 100ms 的等待延迟
                    Thread.sleep(100);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            };
            // 租户不隔离
            if (RosefinchUtil.noTenant()) {
                return executor;
            }
            // 按租户隔离
            return () -> byTenantDirectly(executor, tenantCode);
        } catch (Exception e) {
            logger.error("[ROSEFINCH] 任务执行异常: ", e);
            throw new RuntimeException("任务执行异常");
        }
    }

    @Override
    public boolean stopImmediately(RosefinchInstanceDetailVO instance) {
        if (ObjectUtil.isNull(producerProxy)) {
            throw new RuntimeException("Rosefinch模块未开启");
        }
        return producerProxy.stop(instance.getTaskCode(), instance.getId());
    }
}
