package com.elitesland.cbpl.scheduling.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.scheduling.entity.QScheduleConfigDO;
import com.elitesland.cbpl.scheduling.entity.QScheduleInstanceDO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstanceDeleteParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstancePagingParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstanceQueryParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstanceSaveParamVO;
import com.elitesland.cbpl.scheduling.vo.resp.ScheduleInstancePagingVO;
import com.elitesland.cbpl.scheduling.vo.resp.ScheduleInstanceRespVO;
import com.elitesland.cbpl.tool.db.SqlUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.StringPath;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.PredicateUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2023/09/12
 */
@Component
@RequiredArgsConstructor
public class ScheduleInstanceRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QScheduleConfigDO scheduleConfigDO = QScheduleConfigDO.scheduleConfigDO;
    private static final QScheduleInstanceDO scheduleInstanceDO = QScheduleInstanceDO.scheduleInstanceDO;

    private final QBean<ScheduleInstancePagingVO> scheduleInstancePagingVO = Projections.bean(
            ScheduleInstancePagingVO.class,
            scheduleInstanceDO.id,
            scheduleInstanceDO.masId,
            scheduleInstanceDO.taskCode.coalesce(scheduleConfigDO.taskCode).as("taskCode"),
            scheduleInstanceDO.taskName.coalesce(scheduleConfigDO.taskName).as("taskName"),
            scheduleInstanceDO.instanceStatus,
            scheduleInstanceDO.errorMessage,
            scheduleInstanceDO.startTime,
            scheduleInstanceDO.endTime,
            scheduleInstanceDO.remark,
            scheduleInstanceDO.createUserId,
            scheduleInstanceDO.creator,
            scheduleInstanceDO.createTime,
            scheduleInstanceDO.modifyUserId,
            scheduleInstanceDO.updater,
            scheduleInstanceDO.modifyTime,
            scheduleInstanceDO.deleteFlag
    );

    private Predicate pagingWhere(ScheduleInstancePagingParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleInstanceDO.deleteFlag.eq(0));
        if (ObjectUtil.isNotNull(query.getTaskCodeName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getTaskCodeName());
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCodeName()).or(scheduleConfigDO.taskName.like(likeStr)));
        }
        if (StrUtil.isNotBlank(query.getInstanceStatus())) {
            predicates.add(scheduleInstanceDO.instanceStatus.eq(query.getInstanceStatus()));
        }
        if (StrUtil.isNotBlank(query.getErrorMessage())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getErrorMessage());
            predicates.add(scheduleInstanceDO.errorMessage.like(likeStr));
        }
        if (ObjectUtil.isNotNull(query.getStartTime())) {
            predicates.add(scheduleInstanceDO.startTime.goe(query.getStartTime()));
        }
        if (ObjectUtil.isNotNull(query.getEndTime())) {
            predicates.add(scheduleInstanceDO.endTime.loe(query.getEndTime()));
        }
        if (StrUtil.isNotBlank(query.getRemark())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getRemark());
            predicates.add(scheduleInstanceDO.remark.like(likeStr));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public long scheduleInstanceCountBy(ScheduleInstancePagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(scheduleInstanceDO.id)
                .from(scheduleInstanceDO)
                .leftJoin(scheduleConfigDO).on(scheduleConfigDO.id.eq(scheduleInstanceDO.masId));
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch().size();
    }

    public List<ScheduleInstancePagingVO> scheduleInstancePageBy(ScheduleInstancePagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(scheduleInstancePagingVO)
                .from(scheduleInstanceDO)
                .leftJoin(scheduleConfigDO).on(scheduleConfigDO.id.eq(scheduleInstanceDO.masId));
        query.setPaging(jpaQuery);
        query.fillOrders(jpaQuery, scheduleInstanceDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch();
    }

    private final QBean<ScheduleInstanceRespVO> scheduleInstanceVO = Projections.bean(
            ScheduleInstanceRespVO.class,
            scheduleInstanceDO.id,
            scheduleInstanceDO.masId,
            scheduleInstanceDO.taskCode.coalesce(scheduleConfigDO.taskCode).as("taskCode"),
            scheduleInstanceDO.taskName.coalesce(scheduleConfigDO.taskName).as("taskName"),
            scheduleInstanceDO.instanceStatus,
            scheduleInstanceDO.errorMessage,
            scheduleInstanceDO.startTime,
            scheduleInstanceDO.endTime,
            scheduleInstanceDO.remark,
            scheduleInstanceDO.createUserId,
            scheduleInstanceDO.creator,
            scheduleInstanceDO.createTime,
            scheduleInstanceDO.modifyUserId,
            scheduleInstanceDO.updater,
            scheduleInstanceDO.modifyTime,
            scheduleInstanceDO.deleteFlag
    );

    private Predicate where(ScheduleInstanceQueryParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleInstanceDO.deleteFlag.eq(0));
        if (ObjectUtil.isNotNull(query.getTaskCodeName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getTaskCodeName());
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCodeName()).or(scheduleConfigDO.taskName.like(likeStr)));
        }
        if (StrUtil.isNotBlank(query.getInstanceStatus())) {
            predicates.add(scheduleInstanceDO.instanceStatus.eq(query.getInstanceStatus()));
        }
        if (StrUtil.isNotBlank(query.getErrorMessage())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getErrorMessage());
            predicates.add(scheduleInstanceDO.errorMessage.like(likeStr));
        }
        if (ObjectUtil.isNotNull(query.getStartTime())) {
            predicates.add(scheduleInstanceDO.startTime.goe(query.getStartTime()));
        }
        if (ObjectUtil.isNotNull(query.getEndTime())) {
            predicates.add(scheduleInstanceDO.endTime.loe(query.getEndTime()));
        }
        if (StrUtil.isNotBlank(query.getRemark())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getRemark());
            predicates.add(scheduleInstanceDO.remark.like(likeStr));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public List<ScheduleInstanceRespVO> scheduleInstanceByParam(ScheduleInstanceQueryParamVO query) {
        var jpaQuery = jpaQueryFactory.select(scheduleInstanceVO)
                .from(scheduleInstanceDO)
                .leftJoin(scheduleConfigDO).on(scheduleConfigDO.id.eq(scheduleInstanceDO.masId));
        jpaQuery.where(this.where(query));
        return jpaQuery.fetch();
    }

    public long update(ScheduleInstanceSaveParamVO saveParam) {
        JPAUpdateClause update = jpaQueryFactory.update(scheduleInstanceDO)
                .set(scheduleInstanceDO.endTime, saveParam.getEndTime())
                .set(scheduleInstanceDO.errorMessage, saveParam.getErrorMessage())
                .where(scheduleInstanceDO.id.eq(saveParam.getId()));
        // 实例状态
        if (StrUtil.isNotBlank(saveParam.getInstanceStatus())) {
            update.set(scheduleInstanceDO.instanceStatus, saveParam.getInstanceStatus());
        }
        // 更新备注
        if (StrUtil.isNotBlank(saveParam.getRemark())) {
            update.set(scheduleInstanceDO.remark, saveParam.getRemark());
        }
        return update.execute();
    }

    public ScheduleInstanceRespVO findById(Long instanceId) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleInstanceDO.deleteFlag.eq(0));
        predicates.add(scheduleInstanceDO.id.eq(instanceId));
        var jpaQuery = jpaQueryFactory.select(scheduleInstanceVO)
                .from(scheduleInstanceDO)
                .leftJoin(scheduleConfigDO).on(scheduleConfigDO.id.eq(scheduleInstanceDO.masId));
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return jpaQuery.fetchOne();
    }

    public long delete(ScheduleInstanceDeleteParamVO paramVO) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleInstanceDO.masId.eq(paramVO.getMasId()));
        predicates.add(scheduleInstanceDO.instanceStatus.in(paramVO.getInstanceStatus()));
        if (ObjectUtil.isNotNull(paramVO.getBeforeTime())) {
            predicates.add(scheduleInstanceDO.startTime.lt(paramVO.getBeforeTime()));
        }
        var delete = jpaQueryFactory.delete(scheduleInstanceDO);
        delete.where(ExpressionUtils.allOf(predicates));
        return delete.execute();
    }
}
