package com.elitesland.cbpl.scheduling.service;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.scheduling.convert.ScheduleInstanceConvert;
import com.elitesland.cbpl.scheduling.domain.InstanceStatus;
import com.elitesland.cbpl.scheduling.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.spi.ScheduleCompleteProvider;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstanceSaveParamVO;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.tenant.TenantAsyncSpi;
import com.elitesland.cbpl.tool.tenant.TenantClientSpi;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.skywalking.apm.toolkit.trace.RunnableWrapper;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author eric.hao
 * @since 2023/09/12
 */
@Slf4j
@RequiredArgsConstructor
public class ScheduleExecuteHandler {

    @Resource
    private ScheduleInstanceService scheduleInstanceService;
    @Autowired(required = false)
    private ScheduleCompleteProvider scheduleCompleteProvider;
    @Autowired(required = false)
    private TenantClientSpi tenantClientSpi;
    @Autowired(required = false)
    private TenantAsyncSpi tenantAsyncSpi;
    private final Map<String, CompletableFuture<?>> instanceFutures = new ConcurrentHashMap<>();

    /**
     * 执行任务封装工具类 - CRON
     * <li>记录实例执行情况</li>
     *
     * @param task 任务管理
     * @return 封装方法
     */
    public Runnable runAuto(ScheduledTask task) {
        String tenantCode = TenantSpiUtil.currentTenantCode();
        Runnable executor = () -> runImmediately(task, "自动执行");
        if (tenantClientSpi == null) {
            return RunnableWrapper.of(executor);
        }
        return RunnableWrapper.of(() -> tenantClientSpi.byTenantDirectly(executor, tenantCode));
    }

    /**
     * 执行任务封装工具类 - 立即执行
     * <li>记录实例执行情况</li>
     *
     * @param task 任务管理
     */
    public Long runImmediately(ScheduledTask task) {
        return runImmediately(task, "手动执行");
    }

    public Long runImmediately(ScheduledTask task, String runType) {
        var instance = instanceLog(task, runType);
        updateInstanceStatus(instance, task.getTriggerTask().getRunnable());
        return instance.getId();
    }

    /**
     * 实例日志初始化
     *
     * @param task    任务管理
     * @param runType 执行方式
     * @return 保存参数
     */
    private ScheduleInstanceSaveParamVO instanceLog(ScheduledTask task, String runType) {
        var saveParam = new ScheduleInstanceSaveParamVO();
        saveParam.setMasId(task.getTaskId());
        saveParam.setTaskCode(task.getTaskCode());
        saveParam.setTaskName(task.getTaskName());
        saveParam.setStartTime(LocalDateTime.now());
        saveParam.setInstanceStatus(InstanceStatus.RUNNING.getCode());
        saveParam.setRemark(runType);
        Long instanceId = scheduleInstanceService.save(saveParam);
        saveParam.setId(instanceId);
        return saveParam;
    }

    /**
     * 更新实例执行情况
     *
     * @param instance 实例日志
     * @param runnable 任务执行内容
     */
    private void updateInstanceStatus(ScheduleInstanceSaveParamVO instance, Runnable runnable) {
        // 1. 实例执行前的回调方法
        try {
            if (scheduleCompleteProvider != null) {
                var response = ScheduleInstanceConvert.INSTANCE.saveParamToVO(instance);
                scheduleCompleteProvider.start(response);
            }
        } catch (Exception e) {
            logger.error("[PHOENIX-SCHEDULE] execute complete start error.", e);
        }
        // TODO 考虑使用其他异步处理逻辑
        if (tenantAsyncSpi == null) {
            // 2. 实例执行
            Throwable throwable = null;
            try {
                // 初始状态，按实例正常执行成功处理
                instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
                runnable.run();
            } catch (Exception e) {
                throwable = e;
                instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
                instance.setErrorMessage(ExceptionUtils.formatException(e, 255));
                logger.error("[PHOENIX-SCHEDULE] execute error: {}", ExceptionUtils.formatException(e));
            }

            // 3. 记录实例执行结果
            instance.setEndTime(LocalDateTime.now());
            scheduleInstanceService.update(instance);

            // 4. 实例执行完后的回调方法
            try {
                if (scheduleCompleteProvider != null) {
                    var response = ScheduleInstanceConvert.INSTANCE.saveParamToVO(instance);
                    scheduleCompleteProvider.whenComplete(response, throwable);
                }
            } catch (Exception e) {
                logger.error("[PHOENIX-SCHEDULE] execute complete callback error.", e);
            }
        }
        // 按租户的实现逻辑处理
        else {
            String instanceFutureKey = instanceFutureKey(instance.getId());

            // 2. 实例执行
            CompletableFuture<Void> future = tenantAsyncSpi.runAsync(runnable);
            future.whenComplete((resp, ex) -> tenantAsyncSpi.runAsync(() -> {
                instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
                if (ObjectUtil.isNotNull(ex)) {
                    instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
                    instance.setErrorMessage(ExceptionUtils.formatException(ex, 255));
                }

                // 3. 记录实例执行结果
                instance.setEndTime(LocalDateTime.now());
                scheduleInstanceService.update(instance);
                instanceFutures.remove(instanceFutureKey);

                // 4. 实例执行完后的回调方法
                try {
                    if (scheduleCompleteProvider != null) {
                        var response = ScheduleInstanceConvert.INSTANCE.saveParamToVO(instance);
                        scheduleCompleteProvider.whenComplete(response, ex);
                    }
                } catch (Exception e) {
                    logger.error("[PHOENIX-SCHEDULE] execute complete callback error.", e);
                }
            }));

            // 加入异步实例管理
            if (!instanceFutures.containsKey(instanceFutureKey)) {
                instanceFutures.put(instanceFutureKey, future);
            }
        }
    }

    /**
     * 立即终止实例
     *
     * @param instanceId 实例id
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    public boolean stopImmediately(Long instanceId) {
        try {
            String instanceFutureKey = instanceFutureKey(instanceId);
            CompletableFuture<?> future = instanceFutures.get(instanceFutureKey);
            // 实例不存在，按终止成功返回
            if (ObjectUtil.isEmpty(future)) {
                return true;
            }
            // 立即终止
            boolean stopTag = future.cancel(true);
            // 读取实例
            var instance = scheduleInstanceService.scheduleInstanceById(instanceId);
            // 更新执行结果
            var saveParam = new ScheduleInstanceSaveParamVO();
            saveParam.setId(instanceId);
            saveParam.setEndTime(LocalDateTime.now());
            if (stopTag) {
                instanceFutures.remove(instanceFutureKey);
                saveParam.setInstanceStatus(InstanceStatus.STOP.getCode());
            }
            saveParam.setRemark(instance.getRemark() + "；手动终止，是否终止成功：" + stopTag + "；");
            scheduleInstanceService.update(saveParam);
            return stopTag;
        } catch (Exception e) {
            logger.error("[PHOENIX-SCHEDULE] stop immediately unexpected error.", e);
            // 终止异常，按失败返回
            return false;
        }
    }

    /**
     * 任务主键，按租户区分
     */
    private String instanceFutureKey(Long instanceId) {
        return TenantSpiUtil.currentTenantCode() + instanceId;
    }
}
