package com.elitesland.cbpl.scheduling.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.scheduling.convert.ScheduleConfigConvert;
import com.elitesland.cbpl.scheduling.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.entity.ScheduleConfigDO;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.scheduling.repo.ScheduleConfigRepo;
import com.elitesland.cbpl.scheduling.repo.ScheduleConfigRepoProc;
import com.elitesland.cbpl.scheduling.repo.ScheduleInstanceRepoProc;
import com.elitesland.cbpl.scheduling.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.spi.ScheduleInstanceSpi;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleConfigQueryParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleConfigSaveParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstanceDeleteParamVO;
import com.elitesland.cbpl.scheduling.vo.resp.ScheduleConfigDetailVO;
import com.elitesland.cbpl.scheduling.vo.resp.ScheduleConfigPagingVO;
import com.elitesland.cbpl.scheduling.vo.resp.ScheduleConfigRespVO;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.cron.CronPatternUtils;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_DELETION_CRON;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ScheduleConfigServiceImpl implements ScheduleConfigService {

    private final ScheduleConfigRepo scheduleConfigRepo;
    private final ScheduleConfigRepoProc scheduleConfigRepoProc;
    private final ScheduleInstanceRepoProc scheduleInstanceRepoProc;
    @Autowired(required = false)
    private DefaultSchedulingRegistrar schedulingRegistrar;
    @Autowired(required = false)
    private ScheduleInstanceSpi scheduleInstanceService;

    @Override
    public PagingVO<ScheduleConfigPagingVO> scheduleConfigPageBy(ScheduleConfigPagingParamVO query) {
        long count = scheduleConfigRepoProc.scheduleConfigCountBy(query);
        if (count > 0) {
            var list = scheduleConfigRepoProc.scheduleConfigPageBy(query);
            if (ObjectUtil.isNotNull(scheduleInstanceService)) {
                scheduleInstanceService.instanceWrap(list);
            }
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    @Override
    public List<ScheduledTask> scheduleConfigByParam(ScheduleConfigQueryParamVO query) {
        List<ScheduleConfigRespVO> tasks = scheduleConfigRepoProc.scheduleConfigByParam(query);
        return tasks.stream()
                .map(row -> ScheduledTask.builder()
                        .taskId(row.getId())
                        .taskCode(row.getTaskCode())
                        .taskName(row.getTaskName())
                        .build().setTrigger(row.getClassName(), row.getMethod(), row.getCron())
                ).collect(Collectors.toList());
    }

    @Override
    public ScheduleConfigDetailVO scheduleConfigById(Long id) {
        var scheduleConfigVO = scheduleConfigRepoProc.findById(id);
        if (ObjectUtil.isNull(scheduleConfigVO)) {
            throw PhoenixException.unchecked("Not Found Data");
        }
        return ScheduleConfigConvert.INSTANCE.respToVO(scheduleConfigVO);
    }

    /**
     * 根据任务id，构建Schedule对象
     */
    private ScheduledTask scheduledByTaskId(Long taskId) {
        ScheduleConfigDetailVO schedule = scheduleConfigById(taskId);
        return ScheduledTask.builder()
                .taskId(taskId)
                .taskCode(schedule.getTaskCode())
                .taskName(schedule.getTaskName())
                .build().setTrigger(schedule.getClassName(), schedule.getMethod(), schedule.getCron());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(ScheduleConfigSaveParamVO saveParam) {
        ScheduledTask task = ScheduledTask.builder()
                .taskId(saveParam.getId())
                .taskCode(saveParam.getTaskCode())
                .taskName(saveParam.getTaskName())
                .build().setTrigger(saveParam.getClassName(), saveParam.getMethod(), saveParam.getCron());
        boolean exists = scheduleConfigRepoProc.existsByCode(saveParam.getId(), saveParam.getTaskCode());
        Assert.isFalse(exists, "任务编码已存在");
        // 新增
        if (saveParam.isNew()) {
            ScheduleConfigDO scheduleConfigDO = ScheduleConfigConvert.INSTANCE.saveParamToDO(saveParam);
            scheduleConfigRepo.save(scheduleConfigDO);
            // 如果启用，加入定时任务
            if (scheduleConfigDO.getActiveFlag().equals("1")) {
                addTriggerTask(task);
            }
            return scheduleConfigDO.getId();
        }
        // 修改
        else {
            Optional<ScheduleConfigDO> scheduleConfigDO = scheduleConfigRepo.findById(saveParam.getId());
            if (scheduleConfigDO.isEmpty()) {
                throw PhoenixException.unchecked("Not Found Data");
            }
            ScheduleConfigDO scheduleConfig = scheduleConfigDO.get();
            ScheduleConfigConvert.INSTANCE.saveParamMergeToDO(saveParam, scheduleConfig);
            scheduleConfigRepo.save(scheduleConfig);
            // 如果启用，更新定时任务
            if (scheduleConfig.getActiveFlag().equals("1")) {
                resetTriggerTask(task);
            }
            return scheduleConfig.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updateStatus(Long id, String status) {
        // Update status based on business.
        scheduleConfigRepoProc.updateStatus(id, status);
        // 禁用
        if (status.equals("0")) {
            cancelTriggerTask(scheduledByTaskId(id));
        }
        // 启用
        else if (status.equals("1")) {
            resetTriggerTask(scheduledByTaskId(id));
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long updateDeleteFlag(List<Long> ids) {
        ids.stream().map(this::scheduledByTaskId).forEach(this::cancelTriggerTask);
        return scheduleConfigRepoProc.updateDeleteFlag(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<Long> ids) {
        ids.stream().map(this::scheduledByTaskId).forEach(this::cancelTriggerTask);
        scheduleConfigRepo.deleteAllById(ids);
    }

    @Override
    public Long execute(Long taskId) {
        try {
            // 读取任务配置
            return runImmediately(scheduledByTaskId(taskId));
        } catch (RuntimeException e) {
            throw PhoenixException.unchecked("任务执行失败：" + e.getMessage());
        } catch (Exception e) {
            logger.error("[PHOENIX-SCHEDULE] execute once error: {}", ExceptionUtils.formatException(e));
            throw PhoenixException.unchecked("任务执行异常");
        }
    }

    @Override
    public List<String> preview(Long taskId) {
        ScheduleConfigDetailVO schedule = scheduleConfigById(taskId);
        return CronPatternUtils.preview(schedule.getCron(), 20);
    }

    @Override
    public Set<String> taskCodes() {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            return schedulingRegistrar.taskCodes();
        }
        return Collections.emptySet();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long deleteInstance(ScheduleInstanceDeleteParamVO paramVO, String source) {
        if (!paramVO.isEmpty()) {
            logger.debug("[PHOENIX-SCHEDULE] {} delete instance: {}", source, BeanUtils.toJsonStr(paramVO));
            return scheduleInstanceRepoProc.delete(paramVO);
        }
        return 0L;
    }

    @Override
    public List<String> strategyPreview() {
        return CronPatternUtils.preview(SCHEDULE_DELETION_CRON, 10);
    }

    /**
     * 加入定时任务
     */
    private void addTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.addTriggerTask(task);
        }
    }

    /**
     * 重置定时任务
     */
    private void resetTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.resetTriggerTask(task);
        }
    }

    /**
     * 取消定时任务 - 执行中的任务不强制终止
     */
    private void cancelTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.cancelTriggerTask(task);
        }
    }

    /**
     * 立即执行
     */
    private Long runImmediately(ScheduledTask task) {
        if (ObjectUtil.isNull(schedulingRegistrar)) {
            throw new RuntimeException("Phoenix-Schedule模块未开启");
        }
        return schedulingRegistrar.runImmediately(task);
    }
}
