package com.elitesland.cbpl.scheduling.task;

import com.elitesland.cbpl.scheduling.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.repo.ScheduleConfigRepoProc;
import com.elitesland.cbpl.scheduling.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.spi.SchedulingRegistrarSpi;
import com.elitesland.cbpl.scheduling.util.SchedulingUtil;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleConfigQueryParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstanceDeleteParamVO;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.List;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_DELETION_CRON;
import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_ENABLED;
import static com.elitesland.cbpl.scheduling.domain.InstanceStatus.*;

/**
 * @author eric.hao
 * @since 2024/04/15
 */
@Slf4j
@Component
@RequiredArgsConstructor
@UnicomTag("__InstanceDeletionTask")
public class InstanceDeletionTask implements SchedulingRegistrarSpi {

    private final ScheduleConfigService scheduleConfigService;
    private final ScheduleConfigRepoProc scheduleConfigRepoProc;

    @Override
    public boolean autoRegister() {
        return SCHEDULE_ENABLED;
    }

    @Override
    public ScheduledTask registerTask() {
        return SchedulingUtil.registerTask(this.getClass(), "任务调度实例-定时删除").build()
                .setTrigger(this.getClass().getName(), "deletion", SCHEDULE_DELETION_CRON);
    }

    public void deletion() {
        // 所有未删除的任务
        var tasks = scheduleConfigRepoProc.scheduleConfigByParam(new ScheduleConfigQueryParamVO());
        for (var task : tasks) {
            if (task.noDeletion()) {
                logger.warn("[PHOENIX-SCHEDULE] {} deletion strategy: {}", task.getTaskCode(), task.getDeletionStrategyName());
                continue;
            }
            var deleteParam = new ScheduleInstanceDeleteParamVO();
            deleteParam.setMasId(task.getId());
            // 完成、异常中断、手动终止
            deleteParam.setInstanceStatus(List.of(COMPLETE.getCode(), INTERRUPT.getCode(), STOP.getCode()));
            deleteParam.setBeforeTime(SchedulingUtil.minusDays(task.getDeletionStrategy()));
            scheduleConfigService.deleteInstance(deleteParam, "AUTO");
        }
    }
}
