package com.elitesland.cbpl.scheduling.config;

import com.elitesland.cbpl.scheduling.data.repo.ScheduleConfigRepoProc;
import com.elitesland.cbpl.scheduling.data.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.queue.producer.DefaultQueueProducer;
import com.elitesland.cbpl.scheduling.queue.producer.ScheduleQueueProducer;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.scheduling.registrar.ScheduleInitialize;
import com.elitesland.cbpl.scheduling.registrar.execute.ScheduleExecuteHandler;
import com.elitesland.cbpl.scheduling.registrar.task.InstanceDeletionTask;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.dromara.dynamictp.core.spring.EnableDynamicTp;
import org.dromara.dynamictp.core.support.ThreadPoolBuilder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;

import java.util.concurrent.ScheduledExecutorService;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_CONFIG_PREFIX;
import static com.elitesland.cbpl.scheduling.constant.ThreadConstant.TP_SCHEDULE_POOL_NAME;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@EnableScheduling
@EnableDynamicTp
@Configuration
@EnableConfigurationProperties(SchedulingProperties.class)
@ConditionalOnProperty(prefix = SCHEDULE_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class SchedulingAutoConfiguration {

    @Bean
    public ScheduleExecuteHandler scheduleExecuteHandler() {
        return new ScheduleExecuteHandler();
    }

    @Bean
    public DefaultSchedulingRegistrar defaultSchedulingRegistrar(ScheduleExecuteHandler scheduleExecuteHandler) {
        logger.info("[PHOENIX-SCHEDULE] Scheduling enabled.");
        return new DefaultSchedulingRegistrar(scheduleExecuteHandler);
    }

    @Bean
    public ScheduleInitialize scheduleInitialize(DefaultSchedulingRegistrar schedulingRegistrar) {
        var schedule = new ScheduleInitialize(schedulingRegistrar);
        logger.info("[PHOENIX-SCHEDULE] Initialize finished.");
        return schedule;
    }

    @Bean
    @ConditionalOnClass
    public InstanceDeletionTask instanceDeletionTask(
            ScheduleConfigService scheduleConfigService, ScheduleConfigRepoProc scheduleConfigRepoProc) {
        return new InstanceDeletionTask(scheduleConfigService, scheduleConfigRepoProc);
    }

    @Bean
    @ConditionalOnProperty(prefix = SCHEDULE_CONFIG_PREFIX, name = "queue-solution", havingValue = "default")
    public ScheduleQueueProducer scheduleQueueProducer() {
        logger.info("[PHOENIX-SCHEDULE] Registry DefaultQueueProducer.");
        return new DefaultQueueProducer();
    }

    @Bean(TP_SCHEDULE_POOL_NAME)
    public ScheduledExecutorService phoenixScheduleTp(SchedulingProperties properties) {
        return ThreadPoolBuilder.newBuilder()
                .threadPoolName(TP_SCHEDULE_POOL_NAME)
                .threadFactory(TP_SCHEDULE_POOL_NAME)
                .corePoolSize(properties.getCorePoolSize())
                .maximumPoolSize(properties.getMaximumPoolSize())
                .queueCapacity(properties.getQueueCapacity())
                .rejectedExecutionHandler(properties.getRejectPolicy())
                .buildScheduled();
    }
}
