package com.elitesland.cbpl.scheduling.config;

import com.dtp.core.spring.EnableDynamicTp;
import com.dtp.core.thread.NamedThreadFactory;
import com.elitesland.cbpl.scheduling.data.repo.ScheduleConfigRepoProc;
import com.elitesland.cbpl.scheduling.data.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.queue.producer.DefaultQueueProducer;
import com.elitesland.cbpl.scheduling.queue.producer.ScheduleQueueProducer;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.scheduling.registrar.ScheduleInitialize;
import com.elitesland.cbpl.scheduling.registrar.execute.ScheduleExecuteHandler;
import com.elitesland.cbpl.scheduling.registrar.task.InstanceDeletionTask;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_CONFIG_PREFIX;
import static com.elitesland.cbpl.scheduling.constant.ThreadConstant.TP_SCHEDULE_POOL_NAME;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@EnableScheduling
@EnableDynamicTp
@Configuration
@EnableConfigurationProperties(SchedulingProperties.class)
@ConditionalOnProperty(prefix = SCHEDULE_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class SchedulingAutoConfiguration {

    @Bean
    @ConditionalOnProperty(prefix = SCHEDULE_CONFIG_PREFIX, name = "queue-solution", havingValue = "default", matchIfMissing = true)
    public ScheduleQueueProducer scheduleQueueProducer() {
        logger.info("[PHOENIX-SCHEDULE] Registry DefaultQueueProducer.");
        return new DefaultQueueProducer();
    }

    @Bean
    @ConditionalOnBean(ScheduleQueueProducer.class)
    public ScheduleExecuteHandler scheduleExecuteHandler() {
        return new ScheduleExecuteHandler();
    }

    @Bean
    @ConditionalOnBean(ScheduleExecuteHandler.class)
    public DefaultSchedulingRegistrar defaultSchedulingRegistrar(ScheduleExecuteHandler scheduleExecuteHandler) {
        logger.info("[PHOENIX-SCHEDULE] Scheduling enabled.");
        return new DefaultSchedulingRegistrar(scheduleExecuteHandler);
    }

    @Bean
    @ConditionalOnBean(DefaultSchedulingRegistrar.class)
    public ScheduleInitialize scheduleInitialize(DefaultSchedulingRegistrar schedulingRegistrar) {
        var schedule = new ScheduleInitialize(schedulingRegistrar);
        logger.info("[PHOENIX-SCHEDULE] Initialize finished.");
        return schedule;
    }

    @Bean
    @ConditionalOnClass
    public InstanceDeletionTask instanceDeletionTask(
            ScheduleConfigService scheduleConfigService, ScheduleConfigRepoProc scheduleConfigRepoProc) {
        return new InstanceDeletionTask(scheduleConfigService, scheduleConfigRepoProc);
    }

    @Bean("phoenixScheduleTp")
    public ScheduledExecutorService phoenixScheduleTp(SchedulingProperties properties) {
        return new ScheduledThreadPoolExecutor(
                properties.getCorePoolSize(),
                new NamedThreadFactory(TP_SCHEDULE_POOL_NAME),
                // todo 暂定
                new ThreadPoolExecutor.AbortPolicy()
        );
    }
}
