package com.elitesland.cbpl.scheduling.data.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.scheduling.data.convert.ScheduleConfigConvert;
import com.elitesland.cbpl.scheduling.data.entity.ScheduleConfigDO;
import com.elitesland.cbpl.scheduling.data.repo.ScheduleConfigRepo;
import com.elitesland.cbpl.scheduling.data.repo.ScheduleConfigRepoProc;
import com.elitesland.cbpl.scheduling.data.repo.ScheduleInstanceRepoProc;
import com.elitesland.cbpl.scheduling.data.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigQueryParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigSaveParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleInstanceDeleteParamVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigDetailVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigPagingVO;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.scheduling.registrar.domain.ScheduledTask;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.cron.CronPatternUtils;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_DELETION_CRON;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ScheduleConfigServiceImpl implements ScheduleConfigService {

    private final ScheduleConfigRepo scheduleConfigRepo;
    private final ScheduleConfigRepoProc scheduleConfigRepoProc;
    private final ScheduleInstanceRepoProc scheduleInstanceRepoProc;
    @Autowired(required = false)
    private DefaultSchedulingRegistrar schedulingRegistrar;

    @Override
    public PagingVO<ScheduleConfigPagingVO> scheduleConfigPageBy(ScheduleConfigPagingParamVO query) {
        long count = scheduleConfigRepoProc.scheduleConfigCountBy(query);
        if (count > 0) {
            var list = scheduleConfigRepoProc.scheduleConfigPageBy(query);
            return new PagingVO<>(count, ScheduleConfigConvert.INSTANCE.doToPageVO(list));
        }
        return new PagingVO<>();
    }

    @Override
    public List<ScheduledTask> scheduleConfigByParam(ScheduleConfigQueryParamVO query) {
        var tasks = scheduleConfigRepoProc.scheduleConfigByParam(query);
        return tasks.stream()
                .map(row -> ScheduledTask.builder()
                        .taskId(row.getId())
                        .taskCode(row.getTaskCode()).taskName(row.getTaskName())
                        .className(row.getClassName()).methodName(row.getMethod())
                        .customArgs(row.getCustomArgs())
                        .cron(row.getCron())
                        .build().createTriggerTask()
                ).collect(Collectors.toList());
    }

    @Override
    public ScheduleConfigDetailVO scheduleConfigById(Long id) {
        Optional<ScheduleConfigDO> scheduleConfigDO = scheduleConfigRepo.findById(id);
        if (scheduleConfigDO.isEmpty()) {
            throw PhoenixException.unchecked("Not Found Data");
        }
        return ScheduleConfigConvert.INSTANCE.doToDetailVO(scheduleConfigDO.get());
    }

    /**
     * 根据任务id，构建Schedule对象
     */
    private ScheduledTask scheduledByTaskId(Long taskId) {
        ScheduleConfigDetailVO schedule = scheduleConfigById(taskId);
        return ScheduledTask.builder()
                .taskId(taskId)
                .taskCode(schedule.getTaskCode()).taskName(schedule.getTaskName())
                .className(schedule.getClassName()).methodName(schedule.getMethod())
                .customArgs(schedule.getCustomArgs())
                .cron(schedule.getCron())
                .build().createTriggerTask();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(ScheduleConfigSaveParamVO saveParam) {
        ScheduledTask task = ScheduledTask.builder()
                .taskId(saveParam.getId())
                .taskCode(saveParam.getTaskCode()).taskName(saveParam.getTaskName())
                .className(saveParam.getClassName()).methodName(saveParam.getMethod())
                .customArgs(saveParam.getCustomArgs())
                .cron(saveParam.getCron())
                .build().createTriggerTask();
        boolean exists = scheduleConfigRepoProc.existsByCode(saveParam.getId(), saveParam.getTaskCode());
        Assert.isFalse(exists, "任务编码已存在");
        // 新增
        if (saveParam.isNew()) {
            ScheduleConfigDO scheduleConfigDO = ScheduleConfigConvert.INSTANCE.saveParamToDO(saveParam);
            scheduleConfigRepo.save(scheduleConfigDO);
            // 如果启用，加入定时任务
            if (scheduleConfigDO.getStatus().equals(ActiveStatus.ACTIVE.getCode())) {
                addTriggerTask(task);
            }
            return scheduleConfigDO.getId();
        }
        // 修改
        else {
            Optional<ScheduleConfigDO> scheduleConfigDO = scheduleConfigRepo.findById(saveParam.getId());
            if (scheduleConfigDO.isEmpty()) {
                throw PhoenixException.unchecked("Not Found Data");
            }
            ScheduleConfigDO scheduleConfig = scheduleConfigDO.get();
            ScheduleConfigConvert.INSTANCE.saveParamMergeToDO(saveParam, scheduleConfig);
            scheduleConfigRepo.save(scheduleConfig);
            // 如果启用，更新定时任务
            if (scheduleConfig.getStatus().equals(ActiveStatus.ACTIVE.getCode())) {
                resetTriggerTask(task);
            }
            return scheduleConfig.getId();
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void active(Long id) {
        scheduleConfigRepoProc.updateStatus(id, ActiveStatus.ACTIVE.getCode());
        resetTriggerTask(scheduledByTaskId(id));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void inactive(Long id) {
        scheduleConfigRepoProc.updateStatus(id, ActiveStatus.INACTIVE.getCode());
        cancelTriggerTask(scheduledByTaskId(id));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long updateDeleteFlag(List<Long> ids) {
        ids.stream().map(this::scheduledByTaskId).forEach(this::cancelTriggerTask);
        return scheduleConfigRepoProc.updateDeleteFlag(ids);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<Long> ids) {
        ids.stream().map(this::scheduledByTaskId).forEach(this::cancelTriggerTask);
        scheduleConfigRepo.deleteAllById(ids);
    }

    @Override
    public void execute(Long taskId) {
        try {
            // 读取任务配置
            runImmediately(scheduledByTaskId(taskId));
        } catch (RuntimeException e) {
            throw PhoenixException.unchecked("任务执行失败：" + e.getMessage());
        } catch (Exception e) {
            logger.error("[PHOENIX-SCHEDULE] execute once error: {}", ExceptionUtils.formatException(e));
            throw PhoenixException.unchecked("任务执行异常");
        }
    }

    @Override
    public List<String> preview(Long taskId) {
        ScheduleConfigDetailVO schedule = scheduleConfigById(taskId);
        return CronPatternUtils.preview(schedule.getCron(), 20);
    }

    @Override
    public List<String> previewCron(String cronStr) {
        return CronPatternUtils.preview(cronStr, 20);
    }

    @Override
    public Set<String> taskCodes() {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            return schedulingRegistrar.taskCodes();
        }
        return Collections.emptySet();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public long deleteInstance(ScheduleInstanceDeleteParamVO paramVO, String source) {
        if (!paramVO.isEmpty()) {
            logger.debug("[PHOENIX-SCHEDULE] {} delete instance: {}", source, BeanUtils.toJsonStr(paramVO));
            return scheduleInstanceRepoProc.delete(paramVO);
        }
        return 0L;
    }

    @Override
    public List<String> strategyPreview() {
        return CronPatternUtils.preview(SCHEDULE_DELETION_CRON, 10);
    }

    @Override
    public boolean stopImmediately(String taskCode) {
        return this.stopImmediately(taskCode, true);
    }

    /**
     * 加入定时任务
     */
    private void addTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.addTriggerTask(task);
        }
    }

    /**
     * 重置定时任务
     */
    private void resetTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.resetTriggerTask(task);
        }
    }

    /**
     * 取消定时任务 - 执行中的任务不强制终止
     */
    private void cancelTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.cancelTriggerTask(task);
        }
    }

    /**
     * 立即执行
     */
    private void runImmediately(ScheduledTask task) {
        if (ObjectUtil.isNull(schedulingRegistrar)) {
            throw new RuntimeException("Phoenix-Schedule模块未开启");
        }
        schedulingRegistrar.runImmediately(task);
    }

    /**
     * 取消/终止 任务
     *
     * @param mayInterruptIfRunning 是否立即终止当前任务
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    private boolean stopImmediately(String taskCode, boolean mayInterruptIfRunning) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            return schedulingRegistrar.stopImmediately(taskCode, mayInterruptIfRunning);
        }
        return true;
    }
}
