package com.elitesland.cbpl.scheduling.data.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.scheduling.constant.ScheduleTag;
import com.elitesland.cbpl.scheduling.data.entity.QScheduleConfigDO;
import com.elitesland.cbpl.scheduling.data.entity.ScheduleConfigDO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigQueryParamVO;
import com.elitesland.cbpl.tool.db.SqlUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Component
@RequiredArgsConstructor
public class ScheduleConfigRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QScheduleConfigDO scheduleConfigDO = QScheduleConfigDO.scheduleConfigDO;

    private Predicate pagingWhere(ScheduleConfigPagingParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleConfigDO.deleteFlag.eq(0));
        if (ObjectUtil.isNotNull(query.getTaskCodeName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getTaskCodeName());
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCodeName()).or(scheduleConfigDO.taskName.like(likeStr)));
        }
        if (StrUtil.isNotBlank(query.getTaskName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getTaskName());
            predicates.add(scheduleConfigDO.taskName.like(likeStr));
        }
        if (StrUtil.isNotBlank(query.getTaskCode())) {
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCode()));
        }
        if (StrUtil.isNotBlank(query.getClassName())) {
            predicates.add(scheduleConfigDO.className.eq(query.getClassName()));
        }
        if (StrUtil.isNotBlank(query.getMethod())) {
            predicates.add(scheduleConfigDO.method.eq(query.getMethod()));
        }
        if (StrUtil.isNotBlank(query.getCron())) {
            predicates.add(scheduleConfigDO.cron.eq(query.getCron()));
        }
        if (ObjectUtil.isNotNull(query.getStatus())) {
            predicates.add(scheduleConfigDO.status.eq(query.getStatus()));
        }
        if (StrUtil.isNotBlank(query.getRemark())) {
            predicates.add(scheduleConfigDO.remark.eq(query.getRemark()));
        }
        // tags不为空，则按条件查询
        if (StrUtil.isNotBlank(query.getTags())) {
            predicates.add(scheduleConfigDO.tags.eq(query.getTags()));
        }
        // 为空时，默认查询为 null or SCHEDULE_BUSINESS_TAG
        else {
            predicates.add(scheduleConfigDO.tags.eq(ScheduleTag.SCHEDULE_BUSINESS_TAG).or(scheduleConfigDO.tags.isNull()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public long scheduleConfigCountBy(ScheduleConfigPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(scheduleConfigDO.id)
                .from(scheduleConfigDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch().size();
    }

    public List<ScheduleConfigDO> scheduleConfigPageBy(ScheduleConfigPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(scheduleConfigDO)
                .from(scheduleConfigDO);
        query.setPaging(jpaQuery);
        query.fillOrders(jpaQuery, scheduleConfigDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch();
    }

    private Predicate where(ScheduleConfigQueryParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleConfigDO.deleteFlag.eq(0));
        if (StrUtil.isNotBlank(query.getTaskName())) {
            predicates.add(scheduleConfigDO.taskName.eq(query.getTaskName()));
        }
        if (StrUtil.isNotBlank(query.getTaskCode())) {
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCode()));
        }
        if (ObjectUtil.isNotNull(query.getStatus())) {
            predicates.add(scheduleConfigDO.status.eq(query.getStatus()));
        }
        if (StrUtil.isNotBlank(query.getTags())) {
            predicates.add(scheduleConfigDO.tags.eq(query.getTags()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public List<ScheduleConfigDO> scheduleConfigByParam(ScheduleConfigQueryParamVO query) {
        var jpaQuery = jpaQueryFactory.select(scheduleConfigDO)
                .from(scheduleConfigDO);
        jpaQuery.where(this.where(query));
        return jpaQuery.fetch();
    }

    public long updateDeleteFlag(List<Long> ids) {
        JPAUpdateClause update = jpaQueryFactory.update(scheduleConfigDO)
                .set(scheduleConfigDO.deleteFlag, 1)
                .where(scheduleConfigDO.id.in(ids));
        return update.execute();
    }

    /**
     * 更新状态
     *
     * @param id     业务单据ID
     * @param status 更新状态
     */
    public long updateStatus(Long id, int status) {
        JPAUpdateClause update = jpaQueryFactory.update(scheduleConfigDO)
                .set(scheduleConfigDO.status, status)
                .where(scheduleConfigDO.id.eq(id));
        return update.execute();
    }

    public boolean existsByCode(Long taskId, String taskCode) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleConfigDO.taskCode.eq(taskCode));
        if (ObjectUtil.isNotNull(taskId)) {
            predicates.add(scheduleConfigDO.id.ne(taskId));
        }
        var jpaQuery = jpaQueryFactory.select(scheduleConfigDO.id)
                .from(scheduleConfigDO);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return !jpaQuery.fetch().isEmpty();
    }
}
