package com.elitesland.cbpl.scheduling.data.controller;

import com.elitesland.cbpl.common.base.EnumValues;
import com.elitesland.cbpl.scheduling.constant.DeletionStrategy;
import com.elitesland.cbpl.scheduling.data.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigSaveParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleInstanceDeleteParamVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigDetailVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigPagingVO;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.db.PagingVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Set;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@Api(tags = "定时任务配置")
@RestController
@RequestMapping("/schedule/config")
@AllArgsConstructor
public class ScheduleConfigController {

    private final ScheduleConfigService scheduleConfigService;

    @ApiOperation(value = "分页查询")
    @PostMapping(value = "/list")
    public HttpResult<PagingVO<ScheduleConfigPagingVO>> list(@RequestBody ScheduleConfigPagingParamVO query) {
        logger.info("[PHOENIX-SCHEDULE] query paging param = {}", query);
        return HttpResult.ok(scheduleConfigService.scheduleConfigPageBy(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public HttpResult<ScheduleConfigDetailVO> detail(@PathVariable("id") Long id) {
        logger.info("[PHOENIX-SCHEDULE] query detail by id = {}", id);
        return HttpResult.ok(scheduleConfigService.scheduleConfigById(id));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public HttpResult<Long> save(@RequestBody @Valid ScheduleConfigSaveParamVO saveParam) {
        logger.info("[PHOENIX-SCHEDULE] create/update record, saveParam = {}", saveParam);
        return HttpResult.ok(scheduleConfigService.save(saveParam));
    }

    @ApiOperation(value = "启用")
    @PatchMapping(value = "/{id}/v")
    public HttpResult<Void> active(@PathVariable("id") Long id) {
        logger.info("[PHOENIX-SCHEDULE] active by id = {}", id);
        scheduleConfigService.active(id);
        return HttpResult.ok();
    }

    @ApiOperation(value = "禁用")
    @PatchMapping(value = "/{id}/x")
    public HttpResult<Void> inactive(@PathVariable("id") Long id) {
        logger.info("[PHOENIX-SCHEDULE] inactive by id = {}", id);
        scheduleConfigService.inactive(id);
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量逻辑删除")
    @PutMapping(value = "/delete")
    public HttpResult<Void> updateDeleteFlag(@RequestBody List<Long> ids) {
        logger.info("[PHOENIX-SCHEDULE] batch logic delete by ids = {}", ids);
        scheduleConfigService.updateDeleteFlag(ids);
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量物理删除")
    @DeleteMapping(value = "/delete/physical")
    public HttpResult<Void> delete(@RequestBody List<Long> ids) {
        logger.info("[PHOENIX-SCHEDULE] batch physical delete by ids = {}", ids);
        scheduleConfigService.delete(ids);
        return HttpResult.ok();
    }

    @ApiOperation(value = "立即执行一次")
    @PostMapping(value = "/execute")
    public HttpResult<Void> execute(Long taskId) {
        logger.info("[PHOENIX-SCHEDULE] execute immediately, taskId = {}", taskId);
        scheduleConfigService.execute(taskId);
        return HttpResult.ok();
    }

    /**
     * 列举当前时间后，20个匹配表达式的日期
     */
    @ApiOperation(value = "CRON表达式预览")
    @GetMapping(value = "/cron/preview")
    public HttpResult<List<String>> preview(Long taskId) {
        logger.info("[PHOENIX-SCHEDULE] cron preview, taskId = {}", taskId);
        return HttpResult.ok(scheduleConfigService.preview(taskId));
    }

    /**
     * 列举当前时间后，20个匹配表达式的日期
     */
    @ApiOperation(value = "CRON表达式预览")
    @GetMapping(value = "/cron/preview/expression")
    public HttpResult<List<String>> preview(String cronStr) {
        logger.info("[PHOENIX-SCHEDULE] cron preview expression, cronStr = {}", cronStr);
        return HttpResult.ok(scheduleConfigService.previewCron(cronStr));
    }

    @ApiOperation(value = "所有任务列表")
    @GetMapping(value = "/task/list")
    public HttpResult<Set<String>> taskCodes() {
        logger.info("[PHOENIX-SCHEDULE] query all tenant taskCodes.");
        return HttpResult.ok(scheduleConfigService.taskCodes());
    }

    @ApiOperation(value = "日志删除策略")
    @GetMapping(value = "/task/strategy")
    public HttpResult<List<EnumValues>> strategy() {
        return HttpResult.ok(DeletionStrategy.enumValues());
    }

    @ApiOperation(value = "日志删除策略-CRON表达式预览")
    @GetMapping(value = "/task/strategy/preview")
    public HttpResult<List<String>> strategyPreview() {
        return HttpResult.ok(scheduleConfigService.strategyPreview());
    }

    @ApiOperation(value = "删除实例日志")
    @DeleteMapping(value = "/delete/instance")
    public HttpResult<Long> delete(@Valid @RequestBody ScheduleInstanceDeleteParamVO paramVO) {
        logger.info("[PHOENIX-SCHEDULE] delete instance param = {}", paramVO);
        return HttpResult.ok(scheduleConfigService.deleteInstance(paramVO, "INVOKE"));
    }

    @ApiOperation(value = "终止任务")
    @PostMapping(value = "/stop")
    public HttpResult<Boolean> stop(String taskCode) {
        logger.info("[PHOENIX-SCHEDULE] stop taskCode = {}", taskCode);
        return HttpResult.ok(scheduleConfigService.stopImmediately(taskCode));
    }
}
