package com.elitesland.cbpl.scheduling.util;

import cn.hutool.core.date.DateUtil;
import cn.hutool.json.JSONConfig;
import cn.hutool.json.JSONUtil;
import com.elitesland.cbpl.scheduling.constant.TenantIsolateStrategy;
import com.elitesland.cbpl.scheduling.registrar.domain.ScheduledTask;
import com.elitesland.cbpl.tool.core.date.DateUtils;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import com.elitesland.cbpl.tool.tenant.TenantClientSpi;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;

import java.time.LocalDateTime;
import java.util.Objects;

import static cn.hutool.core.date.DatePattern.NORM_DATE_PATTERN;
import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.TENANT_ISOLATE_STRATEGY;

/**
 * @author eric.hao
 * @since 2024/05/13
 */
public class SchedulingUtil {

    /**
     * 注册个通用的调度任务
     * <li>也可以使用`ScheduledTask.builder()`自定义任务编码等参数</li>
     *
     * @param clazz 调度任务实体类
     */
    public static ScheduledTask.ScheduledTaskBuilder registerTask(Class<?> clazz) {
        return ScheduledTask.builder()
                .taskId(getTaskId(clazz))
                .taskCode(getTaskCode(clazz))
                .className(clazz.getName());
    }

    /**
     * 使用hash绝对值作为任务ID
     */
    public static long getTaskId(Class<?> clazz) {
        // 使用类名和一个固定的包装类生成一个哈希码
        int hashCode = Objects.hash(clazz.getName(), Integer.class);
        return Math.abs(hashCode);
    }

    /**
     * 任务编码
     */
    public static String getTaskCode(Class<?> clazz) {
        UnicomTag anno = clazz.getAnnotation(UnicomTag.class);
        return anno.value();
    }

    /**
     * 获取当前日期的前{days}天的日期
     *
     * @param days 天数
     * @return 前{days}天数的日期 「yyyy-MM-dd 00:00:00」
     */
    public static LocalDateTime minusDays(String days) {
        LocalDateTime today = DateUtils.of(DateUtil.today(), NORM_DATE_PATTERN).toLocalDateTime();
        return today.minusDays(Long.parseLong(days));
    }

    /**
     * 包装参数，反射方法调用
     */
    public static Object parseArgs(String customArgs) {
        return JSONUtil.wrap(customArgs, new JSONConfig());
    }

    /**
     * 是否整合了租户环境
     */
    public static boolean noTenant() {
        // 配置不隔离 or 未对`TenantClientSpi`开发实现类
        return TENANT_ISOLATE_STRATEGY.equals(TenantIsolateStrategy.NONE) || !SpringUtils.isPresent(TenantClientSpi.class);
    }
}
