package com.elitesland.cbpl.scheduling.registrar.execute;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.logging.common.util.LogTraceUtil;
import com.elitesland.cbpl.logging.syslog.util.LogUtil;
import com.elitesland.cbpl.scheduling.constant.InstanceStatus;
import com.elitesland.cbpl.scheduling.data.convert.ScheduleInstanceConvert;
import com.elitesland.cbpl.scheduling.data.service.ScheduleInstanceService;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleInstanceSaveParamVO;
import com.elitesland.cbpl.scheduling.queue.producer.ScheduleQueueProducer;
import com.elitesland.cbpl.scheduling.registrar.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.spi.ScheduleCompleteProvider;
import com.elitesland.cbpl.scheduling.util.SchedulingUtil;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.util.LockUtil;
import com.elitesland.cbpl.tool.core.util.StringUtils;
import com.elitesland.cbpl.tool.tenant.TenantClientSpi;
import com.elitesland.cbpl.tool.tenant.TenantSpiUtil;
import com.lzhpo.tracer.util.TracerUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.skywalking.apm.toolkit.trace.RunnableWrapper;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_CLUSTER;
import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_EXPIRE_DURATION;
import static com.elitesland.cbpl.scheduling.util.SchedulingUtil.taskFutureKey;
import static com.elitesland.cbpl.tool.core.util.StringUtils.DEFAULT_EXCEED_LENGTH;

/**
 * @author eric.hao
 * @since 2023/09/12
 */
@Slf4j
@RequiredArgsConstructor
public class ScheduleExecuteHandler {

    @Resource
    private ScheduleInstanceService scheduleInstanceService;
    @Autowired(required = false)
    private ScheduleQueueProducer scheduleQueueProducer;
    @Autowired(required = false)
    private ScheduleCompleteProvider scheduleCompleteProvider;
    @Autowired(required = false)
    private TenantClientSpi tenantClientSpi;
    private final Map<String, CompletableFuture<?>> instanceFutures = new ConcurrentHashMap<>();

    /**
     * 执行任务封装工具类 - CRON
     * <li>记录实例执行情况</li>
     *
     * @param task 任务管理
     * @return 封装方法
     */
    public Runnable runAuto(ScheduledTask task) {
        Runnable executor = () -> runImmediately(task, "自动执行");
        if (SchedulingUtil.noTenant()) {
            return RunnableWrapper.of(executor);
        }
        String tenantCode = TenantSpiUtil.currentTenantCode();
        return RunnableWrapper.of(() -> tenantClientSpi.byTenantDirectly(executor, tenantCode));
    }

    /**
     * 执行任务封装工具类 - 立即执行
     * <li>记录实例执行情况</li>
     *
     * @param task 任务管理
     */
    public void runImmediately(ScheduledTask task) {
        runImmediately(task, "手动执行");
    }

    public Long runImmediately(ScheduledTask task, String runType) {
        Supplier<Long> supplier = () -> {
            var instance = instanceLog(task, runType);
            scheduleQueueProducer.send(instance, task.getTriggerTask().getRunnable());
            return instance.getId();
        };
        // 单机环境
        if (SCHEDULE_CLUSTER) {
            return supplier.get();
        }
        // 集群环境
        return LockUtil.executeByLock(taskFutureKey(task.getTaskCode()), supplier, Duration.ofSeconds(SCHEDULE_EXPIRE_DURATION));
    }

    /**
     * 实例日志初始化
     *
     * @param task    任务管理
     * @param runType 执行方式
     * @return 保存参数
     */
    private ScheduleInstanceSaveParamVO instanceLog(ScheduledTask task, String runType) {
        var saveParam = new ScheduleInstanceSaveParamVO();
        saveParam.setMasId(task.getTaskId());
        saveParam.setTaskCode(task.getTaskCode());
        saveParam.setTaskName(task.getTaskName());
        saveParam.setStartTime(LocalDateTime.now());
        saveParam.setInstanceStatus(InstanceStatus.READY.getCode());
        saveParam.setRemark(runType);
        Long instanceId = scheduleInstanceService.save(saveParam);
        saveParam.setId(instanceId);
        return saveParam;
    }

    /**
     * 更新实例执行情况
     *
     * @param instance 实例日志
     * @param runnable 任务执行内容
     */
    public void execute(ScheduleInstanceSaveParamVO instance, Runnable runnable) {
        LogTraceUtil.initTraceId();
        LogUtil.info("[CRON][TaskCode] " + instance.getTaskCode() + " 开始执行:");
        // 1. 实例执行前的回调方法
        try {
            if (scheduleCompleteProvider != null) {
                var response = ScheduleInstanceConvert.INSTANCE.saveParamToVO(instance);
                scheduleCompleteProvider.start(response);
            }
        } catch (Exception e) {
            logger.error("[PHOENIX-SCHEDULE] execute complete start error.", e);
        }
        // 2. 实例执行
        Throwable throwable = null;
        try {
            // 更新状态：任务运行中
            instance.setTraceId(TracerUtils.getTraceId());
            instance.setInstanceStatus(InstanceStatus.RUNNING.getCode());
            instance.setExecuteTime(LocalDateTime.now());
            scheduleInstanceService.update(instance);
            // 初始状态，按实例正常执行成功处理
            instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
            runnable.run();
        } catch (Exception e) {
            throwable = e;
            instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
            instance.setErrorMessage(ExceptionUtils.formatException(e, DEFAULT_EXCEED_LENGTH));
            LogUtil.error("[CRON] 执行异常：", e);
        }

        // 3. 记录实例执行结果
        instance.setEndTime(LocalDateTime.now());
        scheduleInstanceService.update(instance);
        LogUtil.info("[CRON][TaskCode] " + instance.getTaskCode() + " 执行结束.");

        // 4. 实例执行完后的回调方法
        try {
            if (scheduleCompleteProvider != null) {
                var response = ScheduleInstanceConvert.INSTANCE.saveParamToVO(instance);
                scheduleCompleteProvider.whenComplete(response, throwable);
            }
        } catch (Exception e) {
            LogUtil.error("[CRON] 实例执行完后的回调方法，执行异常：", e);
        }
        LogTraceUtil.clearTraceId();
    }

    /**
     * 立即终止实例
     *
     * @param instanceId 实例id
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    public boolean stopImmediately(Long instanceId) {
        try {
            String instanceFutureKey = instanceFutureKey(instanceId);
            CompletableFuture<?> future = instanceFutures.get(instanceFutureKey);
            // 实例不存在，按终止成功返回
            if (ObjectUtil.isEmpty(future)) {
                return true;
            }
            // 立即终止
            boolean stopTag = future.cancel(true);
            // 读取实例
            var instance = scheduleInstanceService.scheduleInstanceById(instanceId);
            // 更新执行结果
            var saveParam = new ScheduleInstanceSaveParamVO();
            saveParam.setId(instanceId);
            saveParam.setEndTime(LocalDateTime.now());
            if (stopTag) {
                instanceFutures.remove(instanceFutureKey);
                saveParam.setInstanceStatus(InstanceStatus.STOP.getCode());
            }
            saveParam.setRemark(instance.getRemark() + "；手动终止，是否终止成功：" + stopTag + "；");
            scheduleInstanceService.update(saveParam);
            return stopTag;
        } catch (Exception e) {
            logger.error("[PHOENIX-SCHEDULE] stop immediately unexpected error.", e);
            // 终止异常，按失败返回
            return false;
        }
    }

    /**
     * 任务主键，按租户区分
     */
    private String instanceFutureKey(Long instanceId) {
        // 租户不隔离
        if (SchedulingUtil.noTenant()) {
            return String.valueOf(instanceId);
        }
        return StringUtils.tenantKey(String.valueOf(instanceId));
    }
}
