package com.elitesland.cbpl.scheduling.data.repo;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.rosefinch.data.entity.QRosefinchConfigDO;
import com.elitesland.cbpl.scheduling.constant.ScheduleTag;
import com.elitesland.cbpl.scheduling.data.entity.QScheduleConfigDO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigQueryParamVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigDetailVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigPagingVO;
import com.elitesland.cbpl.tool.db.SqlUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Component
@RequiredArgsConstructor
public class ScheduleConfigRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private static final QRosefinchConfigDO rosefinchConfigDO = QRosefinchConfigDO.rosefinchConfigDO;
    private static final QScheduleConfigDO scheduleConfigDO = QScheduleConfigDO.scheduleConfigDO;

    private final QBean<ScheduleConfigPagingVO> pagingVO = Projections.bean(
            ScheduleConfigPagingVO.class,
            scheduleConfigDO.id,
            scheduleConfigDO.taskName,
            scheduleConfigDO.taskCode,
            scheduleConfigDO.cron,
            scheduleConfigDO.rosefinchId,
            rosefinchConfigDO.className,
            rosefinchConfigDO.method,
            rosefinchConfigDO.methodArgs,
            rosefinchConfigDO.taskTag,
            rosefinchConfigDO.status,
            rosefinchConfigDO.queueStrategy,
            rosefinchConfigDO.deletionStrategy,
            rosefinchConfigDO.tenantId,
            rosefinchConfigDO.remark,
            rosefinchConfigDO.createUserId,
            rosefinchConfigDO.creator,
            rosefinchConfigDO.createTime,
            rosefinchConfigDO.modifyUserId,
            rosefinchConfigDO.updater,
            rosefinchConfigDO.modifyTime,
            rosefinchConfigDO.deleteFlag
    );

    private Predicate pagingWhere(ScheduleConfigPagingParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleConfigDO.deleteFlag.eq(0));
        if (ObjectUtil.isNotNull(query.getTaskCodeName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getTaskCodeName());
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCodeName()).or(scheduleConfigDO.taskName.like(likeStr)));
        }
        if (StrUtil.isNotBlank(query.getTaskName())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getTaskName());
            predicates.add(scheduleConfigDO.taskName.like(likeStr));
        }
        if (StrUtil.isNotBlank(query.getTaskCode())) {
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCode()));
        }
        if (StrUtil.isNotBlank(query.getClassName())) {
            predicates.add(rosefinchConfigDO.className.eq(query.getClassName()));
        }
        if (StrUtil.isNotBlank(query.getMethod())) {
            predicates.add(rosefinchConfigDO.method.eq(query.getMethod()));
        }
        if (StrUtil.isNotBlank(query.getCron())) {
            predicates.add(scheduleConfigDO.cron.eq(query.getCron()));
        }
        if (ObjectUtil.isNotNull(query.getStatus())) {
            predicates.add(rosefinchConfigDO.status.eq(query.getStatus()));
        }
        if (StrUtil.isNotBlank(query.getRemark())) {
            predicates.add(scheduleConfigDO.remark.eq(query.getRemark()));
        }
        // tags不为空，则按条件查询
        if (StrUtil.isNotBlank(query.getTaskTag())) {
            predicates.add(rosefinchConfigDO.taskTag.eq(query.getTaskTag()));
        }
        // 为空时，默认查询为 null or SCHEDULE_BUSINESS_TAG
        else {
            predicates.add(rosefinchConfigDO.taskTag.eq(ScheduleTag.SCHEDULE_BUSINESS_TAG).or(rosefinchConfigDO.taskCode.isNull()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public long scheduleConfigCountBy(ScheduleConfigPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(scheduleConfigDO.id)
                .from(scheduleConfigDO)
                .leftJoin(rosefinchConfigDO).on(rosefinchConfigDO.id.eq(scheduleConfigDO.rosefinchId));
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch().size();
    }

    public List<ScheduleConfigPagingVO> scheduleConfigPageBy(ScheduleConfigPagingParamVO query) {
        var jpaQuery = jpaQueryFactory.select(pagingVO)
                .from(scheduleConfigDO)
                .leftJoin(rosefinchConfigDO).on(rosefinchConfigDO.id.eq(scheduleConfigDO.rosefinchId));
        query.setPaging(jpaQuery);
        query.fillOrders(jpaQuery, scheduleConfigDO);
        jpaQuery.where(this.pagingWhere(query));
        return jpaQuery.fetch();
    }

    private final QBean<ScheduleConfigDetailVO> detalVO = Projections.bean(
            ScheduleConfigDetailVO.class,
            scheduleConfigDO.id,
            scheduleConfigDO.taskName,
            scheduleConfigDO.taskCode,
            scheduleConfigDO.cron,
            scheduleConfigDO.rosefinchId,
            rosefinchConfigDO.className,
            rosefinchConfigDO.method,
            rosefinchConfigDO.methodArgs,
            rosefinchConfigDO.taskTag,
            rosefinchConfigDO.status,
            rosefinchConfigDO.queueStrategy,
            rosefinchConfigDO.deletionStrategy,
            rosefinchConfigDO.tenantId,
            rosefinchConfigDO.remark,
            rosefinchConfigDO.createUserId,
            rosefinchConfigDO.creator,
            rosefinchConfigDO.createTime,
            rosefinchConfigDO.modifyUserId,
            rosefinchConfigDO.updater,
            rosefinchConfigDO.modifyTime,
            rosefinchConfigDO.deleteFlag
    );

    private Predicate where(ScheduleConfigQueryParamVO query) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleConfigDO.deleteFlag.eq(0));
        if (StrUtil.isNotBlank(query.getTaskName())) {
            predicates.add(scheduleConfigDO.taskName.eq(query.getTaskName()));
        }
        if (StrUtil.isNotBlank(query.getTaskCode())) {
            predicates.add(scheduleConfigDO.taskCode.eq(query.getTaskCode()));
        }
        if (ObjectUtil.isNotNull(query.getStatus())) {
            predicates.add(rosefinchConfigDO.status.eq(query.getStatus()));
        }
        if (StrUtil.isNotBlank(query.getTaskTag())) {
            predicates.add(rosefinchConfigDO.taskTag.eq(query.getTaskTag()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    public List<ScheduleConfigDetailVO> scheduleConfigByParam(ScheduleConfigQueryParamVO query) {
        var jpaQuery = jpaQueryFactory.select(detalVO)
                .from(scheduleConfigDO)
                .leftJoin(rosefinchConfigDO).on(rosefinchConfigDO.id.eq(scheduleConfigDO.rosefinchId));
        jpaQuery.where(this.where(query));
        return jpaQuery.fetch();
    }

    public long updateDeleteFlag(List<Long> ids) {
        JPAUpdateClause update = jpaQueryFactory.update(scheduleConfigDO)
                .set(scheduleConfigDO.deleteFlag, 1)
                .where(scheduleConfigDO.id.in(ids));
        return update.execute();
    }

    public boolean existsByCode(Long taskId, String taskCode) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scheduleConfigDO.taskCode.eq(taskCode));
        if (ObjectUtil.isNotNull(taskId)) {
            predicates.add(scheduleConfigDO.id.ne(taskId));
        }
        var jpaQuery = jpaQueryFactory.select(scheduleConfigDO.id)
                .from(scheduleConfigDO);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return !jpaQuery.fetch().isEmpty();
    }

    public ScheduleConfigDetailVO findById(Long taskId) {
        var jpaQuery = jpaQueryFactory.select(detalVO)
                .from(scheduleConfigDO)
                .leftJoin(rosefinchConfigDO).on(rosefinchConfigDO.id.eq(scheduleConfigDO.rosefinchId));
        jpaQuery.where(scheduleConfigDO.id.eq(taskId));
        return jpaQuery.fetchOne();
    }
}
