package com.elitesland.cbpl.scheduling.config;

import com.elitesland.cbpl.rosefinch.data.service.RosefinchConfigService;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchInstanceService;
import com.elitesland.cbpl.rosefinch.registrar.task.InstanceDeletionTask;
import com.elitesland.cbpl.scheduling.constant.ThreadConstant;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.scheduling.registrar.ScheduleInitialize;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.dromara.dynamictp.core.spring.EnableDynamicTp;
import org.dromara.dynamictp.core.support.ThreadPoolBuilder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;

import java.util.concurrent.ScheduledExecutorService;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_CONFIG_PREFIX;
import static com.elitesland.cbpl.scheduling.constant.ThreadConstant.TP_SCHEDULE_POOL_NAME;
import static com.elitesland.cbpl.scheduling.constant.ThreadConstant.TP_SCHEDULE_PREFIX;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@EnableScheduling
@EnableDynamicTp
@Configuration
@EnableConfigurationProperties(SchedulingProperties.class)
@ConditionalOnProperty(prefix = SCHEDULE_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class SchedulingAutoConfiguration {

    @Bean
    @ConditionalOnBean(RosefinchConfigService.class)
    public DefaultSchedulingRegistrar defaultSchedulingRegistrar(RosefinchConfigService rosefinchConfigService) {
        logger.info("[SCHEDULE] Scheduling enabled.");
        return new DefaultSchedulingRegistrar(rosefinchConfigService);
    }

    @Bean
    public ScheduleInitialize scheduleInitialize(DefaultSchedulingRegistrar schedulingRegistrar) {
        var schedule = new ScheduleInitialize(schedulingRegistrar);
        logger.info("[SCHEDULE] Initialize finished.");
        return schedule;
    }

    @Bean
    @ConditionalOnClass({RosefinchConfigService.class, RosefinchInstanceService.class})
    public InstanceDeletionTask instanceDeletionTask(
            RosefinchConfigService rosefinchConfigService, RosefinchInstanceService rosefinchInstanceService) {
        return new InstanceDeletionTask(rosefinchConfigService, rosefinchInstanceService);
    }

    @Bean(ThreadConstant.TP_SCHEDULE_POOL_NAME)
    public ScheduledExecutorService phoenixScheduleTp(SchedulingProperties properties) {
        return ThreadPoolBuilder.newBuilder()
                .threadPoolName(TP_SCHEDULE_POOL_NAME)
                .threadFactory(TP_SCHEDULE_PREFIX)
                .corePoolSize(properties.getCorePoolSize())
                .maximumPoolSize(properties.getMaximumPoolSize())
                .queueCapacity(properties.getQueueCapacity())
                .rejectedExecutionHandler(properties.getRejectPolicy())
                .buildScheduled();
    }
}
