package com.elitesland.cbpl.scheduling.registrar;

import com.elitesland.cbpl.scheduling.registrar.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.registrar.execute.ScheduleExecuteHandler;
import com.elitesland.cbpl.scheduling.util.SchedulingUtil;
import com.elitesland.cbpl.tool.core.util.StringUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.SchedulingException;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;

import javax.annotation.Resource;
import javax.validation.constraints.NotNull;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@RequiredArgsConstructor
public class DefaultSchedulingRegistrar implements SchedulingConfigurer {

    private final ScheduleExecuteHandler scheduleExecuteHandler;
    private ScheduledTaskRegistrar taskRegistrar;
    private final Map<String, ScheduledFuture<?>> taskFutures = new ConcurrentHashMap<>();

    @Resource
    private ScheduledExecutorService phoenixScheduleTp;

    @Override
    public void configureTasks(@NotNull ScheduledTaskRegistrar taskRegistrar) {
        taskRegistrar.setScheduler(phoenixScheduleTp);
        this.taskRegistrar = taskRegistrar;
    }

    /**
     * 立即执行一次
     */
    public void runImmediately(ScheduledTask task) {
        scheduleExecuteHandler.runImmediately(task);
    }

    /**
     * 添加任务
     */
    public void addTriggerTask(ScheduledTask task) {
        String taskFutureKey = SchedulingUtil.taskFutureKey(task.getTaskCode());
        if (taskFutures.containsKey(taskFutureKey)) {
            throw new SchedulingException("[PHOENIX-SCHEDULE] taskCode[" + task.getTaskCode() + "] was added.");
        }
        TaskScheduler scheduler = taskRegistrar.getScheduler();
        ScheduledFuture<?> future = scheduler.schedule(scheduleExecuteHandler.runAuto(task), task.getTriggerTask().getTrigger());
        taskFutures.put(taskFutureKey, future);
        logger.info("[PHOENIX-SCHEDULE] add taskCode[{}] success.", task.getTaskCode());
    }

    /**
     * 取消任务
     */
    public void cancelTriggerTask(ScheduledTask task) {
        if (!this.hasTask(task.getTaskCode())) {
            logger.info("[PHOENIX-SCHEDULE] taskCode[{}] not exist to cancel.", task.getTaskCode());
            return;
        }
        boolean cancelTag = this.stopImmediately(task.getTaskCode(), false);
        logger.info("[PHOENIX-SCHEDULE] taskCode[{}] canceled {}.", task.getTaskCode(), cancelTag);
    }

    /**
     * 终止任务
     */
    public boolean stopTriggerTask(ScheduledTask task) {
        // 任务不存在，按终止成功返回
        if (!this.hasTask(task.getTaskCode())) {
            logger.warn("[PHOENIX-SCHEDULE] taskCode[{}] not exist to stop.", task.getTaskCode());
            return true;
        }
        boolean stopTag = this.stopImmediately(task.getTaskCode(), true);
        logger.info("[PHOENIX-SCHEDULE] taskCode[{}] stopped {}.", task.getTaskCode(), stopTag);
        return stopTag;
    }

    /**
     * 取消/终止 任务
     *
     * @param mayInterruptIfRunning 是否立即终止当前任务
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    public boolean stopImmediately(String taskCode, boolean mayInterruptIfRunning) {
        String taskFutureKey = SchedulingUtil.taskFutureKey(taskCode);
        ScheduledFuture<?> future = taskFutures.get(taskFutureKey);
        boolean cancelTag = future.cancel(mayInterruptIfRunning);
        if (cancelTag) {
            taskFutures.remove(taskFutureKey);
        }
        return cancelTag;
    }

    /**
     * 终止实例【仅终止实例，不终止任务】
     *
     * @param instanceId 实例ID
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    public boolean stopTriggerInstance(Long instanceId) {
        boolean stopTag = scheduleExecuteHandler.stopImmediately(instanceId);
        logger.info("[PHOENIX-SCHEDULE] instanceId[{}] stopped {}.", instanceId, stopTag);
        return stopTag;
    }

    /**
     * 重置任务
     */
    public void resetTriggerTask(ScheduledTask task) {
        this.cancelTriggerTask(task);
        this.addTriggerTask(task);
    }

    /**
     * 任务编号
     */
    public Set<String> taskCodes() {
        return taskFutures.keySet();
    }

    /**
     * 是否存在任务
     */
    public boolean hasTask(String taskCode) {
        return this.taskFutures.containsKey(SchedulingUtil.taskFutureKey(taskCode));
    }

    /**
     * 任务调度是否已经初始化完成
     */
    public boolean initializeCompleted() {
        return this.taskRegistrar != null && this.taskRegistrar.getScheduler() != null;
    }
}
