package com.elitesland.cbpl.scheduling.controller;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.cbpl.scheduling.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleConfigSaveParamVO;
import com.elitesland.cbpl.scheduling.vo.resp.ScheduleConfigDetailVO;
import com.elitesland.cbpl.scheduling.vo.resp.ScheduleConfigPagingVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@Api(tags = "定时任务配置")
@RestController
@RequestMapping("/schedule/config")
@AllArgsConstructor
public class ScheduleConfigController {

    private final ScheduleConfigService scheduleConfigService;

    @ApiOperation(value = "分页查询")
    @PostMapping(value = "/list")
    public ApiResult<PagingVO<ScheduleConfigPagingVO>> list(@RequestBody ScheduleConfigPagingParamVO query) {
        logger.info("[PHOENIX-SCHEDULE] query paging param = {}", query);
        return ApiResult.ok(scheduleConfigService.scheduleConfigPageBy(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public ApiResult<ScheduleConfigDetailVO> detail(@PathVariable("id") Long id) {
        logger.info("[PHOENIX-SCHEDULE] query detail by id = {}", id);
        return ApiResult.ok(scheduleConfigService.scheduleConfigById(id));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public ApiResult<Long> save(@RequestBody ScheduleConfigSaveParamVO saveParam) {
        logger.info("[PHOENIX-SCHEDULE] create/update record, saveParam = {}", saveParam);
        return ApiResult.ok(scheduleConfigService.save(saveParam));
    }

    @ApiOperation(value = "启用")
    @PatchMapping(value = "/{id}/v")
    public ApiResult<Void> active(@PathVariable("id") Long id) {
        logger.info("[PHOENIX-SCHEDULE] active by id = {}", id);
        scheduleConfigService.updateStatus(id, "1");
        return ApiResult.ok();
    }

    @ApiOperation(value = "禁用")
    @PatchMapping(value = "/{id}/x")
    public ApiResult<Void> inactive(@PathVariable("id") Long id) {
        logger.info("[PHOENIX-SCHEDULE] inactive by id = {}", id);
        scheduleConfigService.updateStatus(id, "0");
        return ApiResult.ok();
    }

    @ApiOperation(value = "批量逻辑删除")
    @PutMapping(value = "/delete")
    public ApiResult<Void> updateDeleteFlag(@RequestBody List<Long> ids) {
        logger.info("[PHOENIX-SCHEDULE] batch logic delete by ids = {}", ids);
        scheduleConfigService.updateDeleteFlag(ids);
        return ApiResult.ok();
    }

    @ApiOperation(value = "立即执行一次")
    @PostMapping(value = "/execute")
    public ApiResult<Void> execute(Long taskId) {
        logger.info("[PHOENIX-SCHEDULE] execute immediately, taskId = {}", taskId);
        scheduleConfigService.execute(taskId);
        return ApiResult.ok();
    }

    /**
     * 列举当前时间后，20个匹配表达式的日期
     */
    @ApiOperation(value = "CRON表达式预览")
    @GetMapping(value = "/cron/preview")
    public ApiResult<List<String>> preview(Long taskId) {
        logger.info("[PHOENIX-SCHEDULE] cron preview, taskId = {}", taskId);
        return ApiResult.ok(scheduleConfigService.preview(taskId));
    }
}
