package com.elitesland.cbpl.scheduling.registrar;

import cn.hutool.core.collection.ConcurrentHashSet;
import com.elitesland.cbpl.scheduling.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.service.ScheduleExecuteHandler;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.scheduling.SchedulingException;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@RequiredArgsConstructor
public class DefaultSchedulingRegistrar implements SchedulingConfigurer {

    private final ScheduleExecuteHandler scheduleExecuteHandler;
    private ScheduledTaskRegistrar taskRegistrar;
    private Set<ScheduledFuture<?>> scheduledFutures;
    private final Map<String, ScheduledFuture<?>> taskFutures = new ConcurrentHashMap<>();

    @Override
    public void configureTasks(@NotNull ScheduledTaskRegistrar taskRegistrar) {
        this.taskRegistrar = taskRegistrar;
    }

    private Set<ScheduledFuture<?>> getScheduledFutures() {
        if (scheduledFutures == null) {
            scheduledFutures = new ConcurrentHashSet<>();
        }
        return scheduledFutures;
    }

    /**
     * 立即执行一次
     */
    public void runImmediately(ScheduledTask task) {
        scheduleExecuteHandler.runImmediately(task);
    }

    /**
     * 添加任务
     */
    public void addTriggerTask(ScheduledTask task) {
        if (taskFutures.containsKey(task.getTaskCode())) {
            throw new SchedulingException("[PHOENIX-SCHEDULE] taskCode[" + task.getTaskCode() + "] was added.");
        }
        TaskScheduler scheduler = taskRegistrar.getScheduler();
        ScheduledFuture<?> future = scheduler.schedule(scheduleExecuteHandler.runAuto(task), task.getTriggerTask().getTrigger());
        getScheduledFutures().add(future);
        taskFutures.put(task.getTaskCode(), future);
        logger.info("[PHOENIX-SCHEDULE] add taskCode[{}] success.", task.getTaskCode());
    }

    /**
     * 取消任务
     */
    public void cancelTriggerTask(ScheduledTask task) {
        if (!hasTask(task.getTaskCode())) {
            logger.info("[PHOENIX-SCHEDULE] taskCode[{}] not exist to cancel.", task.getTaskCode());
            return;
        }
        boolean cancelTag = stopImmediately(task.getTaskCode(), false);
        logger.info("[PHOENIX-SCHEDULE] taskCode[{}] canceled {}.", task.getTaskCode(), cancelTag);
    }

    /**
     * 终止任务
     */
    public boolean stopTriggerTask(String taskCode, Long instanceId) {
        // 任务不存在，按终止成功返回
        if (!hasTask(taskCode)) {
            logger.warn("[PHOENIX-SCHEDULE] taskCode[{}] not exist to stop.", taskCode);
            return true;
        }
        boolean stopTag = stopImmediately(taskCode, true);
        scheduleExecuteHandler.stopImmediately(instanceId, stopTag);
        logger.info("[PHOENIX-SCHEDULE] taskCode[{}] stopped {}.", taskCode, stopTag);
        return stopTag;
    }

    /**
     * 取消/终止 任务
     *
     * @param mayInterruptIfRunning 是否立即终止当前任务
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    private boolean stopImmediately(String taskCode, boolean mayInterruptIfRunning) {
        ScheduledFuture<?> future = taskFutures.get(taskCode);
        boolean cancelTag = future.cancel(mayInterruptIfRunning);
        taskFutures.remove(taskCode);
        getScheduledFutures().remove(future);
        return cancelTag;
    }

    /**
     * 重置任务
     */
    public void resetTriggerTask(ScheduledTask task) {
        cancelTriggerTask(task);
        addTriggerTask(task);
    }

    /**
     * 任务编号
     */
    public Set<String> taskCodes() {
        return taskFutures.keySet();
    }

    /**
     * 是否存在任务
     */
    public boolean hasTask(String taskCode) {
        return this.taskFutures.containsKey(taskCode);
    }

    /**
     * 任务调度是否已经初始化完成
     */
    public boolean initializeCompleted() {
        return this.taskRegistrar != null && this.taskRegistrar.getScheduler() != null;
    }
}
