package com.elitesland.cbpl.scheduling.service;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.cloudt.tenant.TenantWrapper;
import com.elitesland.cbpl.scheduling.convert.ScheduleInstanceConvert;
import com.elitesland.cbpl.scheduling.domain.InstanceStatus;
import com.elitesland.cbpl.scheduling.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.spi.ScheduleCompleteProvider;
import com.elitesland.cbpl.scheduling.vo.param.ScheduleInstanceSaveParamVO;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.concurrent.CompletableFuture;

/**
 * @author eric.hao
 * @since 2023/09/12
 */
@Slf4j
@RequiredArgsConstructor
public class ScheduleExecuteHandler {

    @Resource
    private TenantWrapper tenantWrapper;
    @Resource
    private ScheduleInstanceService scheduleInstanceService;
    @Autowired(required = false)
    private ScheduleCompleteProvider scheduleCompleteProvider;

    /**
     * 执行任务封装工具类 - CRON
     * <li>记录实例执行情况</li>
     *
     * @param task 任务管理
     * @return 封装方法
     */
    public Runnable runAuto(ScheduledTask task) {
        return () -> runImmediately(task, "自动执行");
    }

    /**
     * 执行任务封装工具类 - 立即执行
     * <li>记录实例执行情况</li>
     *
     * @param task 任务管理
     */
    public void runImmediately(ScheduledTask task) {
        runImmediately(task, "手动执行");
    }

    public void runImmediately(ScheduledTask task, String runType) {
        var instance = instanceLog(task, runType);
        updateInstanceStatus(instance, task.getTriggerTask().getRunnable());
    }

    /**
     * 实例日志初始化
     *
     * @param task    任务管理
     * @param runType 执行方式
     * @return 保存参数
     */
    private ScheduleInstanceSaveParamVO instanceLog(ScheduledTask task, String runType) {
        var saveParam = new ScheduleInstanceSaveParamVO();
        saveParam.setMasId(task.getTaskId());
        saveParam.setTaskCode(task.getTaskCode());
        saveParam.setTaskName(task.getTaskName());
        saveParam.setStartTime(LocalDateTime.now());
        saveParam.setInstanceStatus(InstanceStatus.RUNNING.getCode());
        saveParam.setRemark(runType);
        Long instanceId = scheduleInstanceService.save(saveParam);
        saveParam.setId(instanceId);
        return saveParam;
    }

    /**
     * 更新实例执行情况
     *
     * @param instance 实例日志
     * @param runnable 任务执行内容
     */
    private void updateInstanceStatus(ScheduleInstanceSaveParamVO instance, Runnable runnable) {
        CompletableFuture<Void> future = tenantWrapper.runAsync(runnable);
        future.whenComplete((resp, ex) -> {
            instance.setInstanceStatus(InstanceStatus.COMPLETE.getCode());
            if (ObjectUtil.isNotNull(ex)) {
                instance.setInstanceStatus(InstanceStatus.INTERRUPT.getCode());
                instance.setErrorMessage(ExceptionUtils.formatException(ex, 255));
            }
            instance.setEndTime(LocalDateTime.now());
            scheduleInstanceService.update(instance);
            try {
                if (scheduleCompleteProvider != null) {
                    var response = ScheduleInstanceConvert.INSTANCE.saveParamToVO(instance);
                    scheduleCompleteProvider.whenComplete(response, ex);
                }
            } catch (Exception e) {
                logger.error("[PHOENIX-SCHEDULE] execute complete callback error.", e);
            }
        });
    }

    /**
     * 立即终止实例
     *
     * @param instanceId 实例id
     * @param stopTag    如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    public void stopImmediately(Long instanceId, boolean stopTag) {
        try {
            // 读取实例
            var instance = scheduleInstanceService.scheduleInstanceById(instanceId);
            // 更新执行结果
            var saveParam = new ScheduleInstanceSaveParamVO();
            saveParam.setId(instanceId);
            saveParam.setEndTime(LocalDateTime.now());
            saveParam.setInstanceStatus(InstanceStatus.STOP.getCode());
            saveParam.setRemark(instance.getRemark() + " 手动终止，是否终止成功：" + stopTag + "；");
            scheduleInstanceService.update(saveParam);
        } catch (Exception e) {
            logger.error("[PHOENIX-SCHEDULE] stop immediately unexpected error.", e);
        }
    }
}
