package com.elitesland.cbpl.scheduling.data.controller;

import com.elitesland.cbpl.scheduling.data.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigSaveParamVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigDetailVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigPagingVO;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.db.PagingVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Set;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@Api(tags = "定时任务配置")
@RestController
@RequestMapping("/schedule/config")
@AllArgsConstructor
public class ScheduleConfigController {

    private final ScheduleConfigService scheduleConfigService;

    @ApiOperation(value = "分页查询")
    @PostMapping(value = "/list")
    public HttpResult<PagingVO<ScheduleConfigPagingVO>> list(@RequestBody ScheduleConfigPagingParamVO query) {
        logger.info("[SCHEDULE] query paging param = {}", query);
        return HttpResult.ok(scheduleConfigService.scheduleConfigPageBy(query));
    }

    @ApiOperation(value = "明细查询")
    @GetMapping(value = "/detail/{id}")
    public HttpResult<ScheduleConfigDetailVO> detail(@PathVariable("id") Long id) {
        logger.info("[SCHEDULE] query detail by id = {}", id);
        return HttpResult.ok(scheduleConfigService.scheduleConfigById(id));
    }

    @ApiOperation(value = "新增或修改")
    @PostMapping(value = "/save")
    public HttpResult<Long> save(@RequestBody @Valid ScheduleConfigSaveParamVO saveParam) {
        logger.info("[SCHEDULE] create/update record, saveParam = {}", saveParam);
        return HttpResult.ok(scheduleConfigService.save(saveParam));
    }

    @ApiOperation(value = "启用")
    @PatchMapping(value = "/{id}/v")
    public HttpResult<Void> active(@PathVariable("id") Long id) {
        logger.info("[SCHEDULE] active by id = {}", id);
        scheduleConfigService.active(id);
        return HttpResult.ok();
    }

    @ApiOperation(value = "禁用")
    @PatchMapping(value = "/{id}/x")
    public HttpResult<Void> inactive(@PathVariable("id") Long id) {
        logger.info("[SCHEDULE] inactive by id = {}", id);
        scheduleConfigService.inactive(id);
        return HttpResult.ok();
    }

    @ApiOperation(value = "批量物理删除")
    @DeleteMapping(value = "/delete/physical")
    public HttpResult<Void> delete(@RequestBody List<Long> ids) {
        logger.info("[SCHEDULE] batch physical delete by ids = {}", ids);
        scheduleConfigService.delete(ids);
        return HttpResult.ok();
    }

    @ApiOperation(value = "立即执行一次")
    @PostMapping(value = "/execute")
    public HttpResult<Void> execute(Long taskId) {
        logger.info("[SCHEDULE] execute immediately, taskId = {}", taskId);
        scheduleConfigService.execute(taskId);
        return HttpResult.ok();
    }

    @ApiOperation(value = "移除并终止任务")
    @PostMapping(value = "/remove")
    public HttpResult<Boolean> stop(String taskCode) {
        logger.info("[SCHEDULE] stop taskCode = {}", taskCode);
        return HttpResult.ok(scheduleConfigService.stopImmediately(taskCode));
    }

    /**
     * 列举当前时间后，20个匹配表达式的日期
     */
    @ApiOperation(value = "CRON表达式预览")
    @GetMapping(value = "/cron/preview")
    public HttpResult<List<String>> preview(Long taskId) {
        logger.info("[SCHEDULE] cron preview, taskId = {}", taskId);
        return HttpResult.ok(scheduleConfigService.preview(taskId));
    }

    @ApiOperation(value = "所有任务列表")
    @GetMapping(value = "/task/list")
    public HttpResult<Set<String>> taskCodes() {
        logger.info("[SCHEDULE] query all tenant taskCodes.");
        return HttpResult.ok(scheduleConfigService.taskCodes());
    }
}
