package com.elitesland.cbpl.rosefinch.registrar.task;

import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.logging.common.constant.DeletionStrategy;
import com.elitesland.cbpl.rosefinch.common.constant.InstanceStatus;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchConfigService;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchInstanceService;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchConfigQueryParamVO;
import com.elitesland.cbpl.rosefinch.data.vo.param.RosefinchInstanceDeleteParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigSaveParamVO;
import com.elitesland.cbpl.scheduling.spi.SchedulingDeletionSpi;
import com.elitesland.cbpl.scheduling.spi.SchedulingRegistrarSpi;
import com.elitesland.cbpl.scheduling.util.SchedulingUtil;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import static com.elitesland.cbpl.rosefinch.config.RosefinchProperties.*;
import static com.elitesland.cbpl.scheduling.util.SchedulingParamUtil.deletionParamInit;

/**
 * @author eric.hao
 * @since 2024/04/15
 */
@Slf4j
@RequiredArgsConstructor
@UnicomTag("__InstanceDeletionTask")
public class InstanceDeletionTask implements SchedulingRegistrarSpi, SchedulingDeletionSpi {

    private final RosefinchConfigService rosefinchConfigService;
    private final RosefinchInstanceService rosefinchInstanceService;

    @Override
    public boolean autoRegister() {
        return ROSEFINCH_ENABLED;
    }

    @Override
    public ScheduleConfigSaveParamVO registerTask() {
        return deletionParamInit(this.getClass(), "任务中心实例-定时删除", ROSEFINCH_DELETION_CRON);
    }

    @Override
    public void deletion() {
        // 所有未删除的任务
        var tasks = rosefinchConfigService.rosefinchConfigByParam(new RosefinchConfigQueryParamVO());
        for (var task : tasks) {
            // 已禁用的任务，按默认天数删除
            if (task.getStatus().equals(ActiveStatus.INACTIVE.getCode())) {
                deleteInstance(task.getTaskCode(), ROSEFINCH_DELETION_DAYS);
                continue;
            }
            // 启用状态的任务，按任务配置，删除实例日志
            deleteInstance(task.getTaskCode(), task.getDeletionStrategy());
        }
    }

    private void deleteInstance(String taskCode, String deletionStrategy) {
        // 配置为不删除
        if (DeletionStrategy.NO_DELETION.getDays().equals(deletionStrategy)) {
            logger.warn("[ROSEFINCH] {} deletion strategy: {}", taskCode, deletionStrategy);
            return;
        }
        var deleteParam = new RosefinchInstanceDeleteParamVO();
        deleteParam.setTaskCode(taskCode);
        deleteParam.setInstanceStatus(InstanceStatus.deletionStatus());
        deleteParam.setBeforeTime(SchedulingUtil.minusDays(deletionStrategy));
        rosefinchInstanceService.deleteInstance(deleteParam, "AUTO");
    }
}