package com.elitesland.cbpl.scheduling.config;

import com.dtp.core.thread.NamedThreadFactory;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchConfigService;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchInstanceService;
import com.elitesland.cbpl.rosefinch.registrar.task.InstanceDeletionTask;
import com.elitesland.cbpl.scheduling.constant.ThreadConstant;
import com.elitesland.cbpl.scheduling.data.repo.ScheduleConfigRepoProc;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.scheduling.registrar.ScheduleInitialize;
import com.elitesland.cbpl.scheduling.registrar.service.ScheduleStatusService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import net.javacrumbs.shedlock.core.LockProvider;
import net.javacrumbs.shedlock.provider.redis.spring.RedisLockProvider;
import net.javacrumbs.shedlock.spring.annotation.EnableSchedulerLock;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.EnableScheduling;

import java.util.Objects;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.SCHEDULE_CONFIG_PREFIX;
import static com.elitesland.cbpl.scheduling.constant.ThreadConstant.TP_SCHEDULE_POOL_NAME;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@EnableScheduling
@Configuration
@EnableConfigurationProperties(SchedulingProperties.class)
@EnableSchedulerLock(defaultLockAtMostFor = "PT2M") // 默认持有锁的最长2分钟
@ConditionalOnProperty(prefix = SCHEDULE_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class SchedulingAutoConfiguration {

    @Bean
    @ConditionalOnClass
    public LockProvider lockProvider(RedisTemplate<?, ?> redisTemplate) {
        return new RedisLockProvider(Objects.requireNonNull(redisTemplate.getConnectionFactory()));
    }

    @Bean
    public DefaultSchedulingRegistrar defaultSchedulingRegistrar() {
        logger.info("[SCHEDULE] Scheduling enabled.");
        return new DefaultSchedulingRegistrar();
    }

    @Bean
    public ScheduleInitialize scheduleInitialize(DefaultSchedulingRegistrar schedulingRegistrar) {
        var schedule = new ScheduleInitialize(schedulingRegistrar);
        logger.info("[SCHEDULE] Initialize finished.");
        return schedule;
    }

    @Bean
    @ConditionalOnClass({RosefinchConfigService.class, RosefinchInstanceService.class})
    public InstanceDeletionTask instanceDeletionTask(
            RosefinchConfigService rosefinchConfigService, RosefinchInstanceService rosefinchInstanceService) {
        return new InstanceDeletionTask(rosefinchConfigService, rosefinchInstanceService);
    }

    @Bean(ThreadConstant.TP_SCHEDULE_POOL_NAME)
    public ScheduledExecutorService phoenixScheduleTp(SchedulingProperties properties) {
//        return ThreadPoolBuilder.newBuilder()
//                .threadPoolName(TP_SCHEDULE_POOL_NAME)
//                .threadFactory(TP_SCHEDULE_PREFIX)
//                .corePoolSize(properties.getCorePoolSize())
//                .maximumPoolSize(properties.getMaximumPoolSize())
//                .queueCapacity(properties.getQueueCapacity())
//                .rejectedExecutionHandler(properties.getRejectPolicy())
//                .buildScheduled();
        return new ScheduledThreadPoolExecutor(
                properties.getCorePoolSize(),
                new NamedThreadFactory(TP_SCHEDULE_POOL_NAME),
                // todo 暂定
                new ThreadPoolExecutor.AbortPolicy()
        );
    }

    @Bean
    @ConditionalOnClass({ScheduleConfigRepoProc.class})
    public ScheduleStatusService scheduleStatusService(ScheduleConfigRepoProc scheduleConfigRepoProc) {
        return new ScheduleStatusService(scheduleConfigRepoProc);
    }
}
