package com.elitesland.cbpl.scheduling.data.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.common.constant.ActiveStatus;
import com.elitesland.cbpl.rosefinch.data.service.RosefinchConfigService;
import com.elitesland.cbpl.scheduling.data.convert.ScheduleConfigConvert;
import com.elitesland.cbpl.scheduling.data.entity.ScheduleConfigDO;
import com.elitesland.cbpl.scheduling.data.repo.ScheduleConfigRepo;
import com.elitesland.cbpl.scheduling.data.repo.ScheduleConfigRepoProc;
import com.elitesland.cbpl.scheduling.data.service.ScheduleConfigService;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigPagingParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigQueryParamVO;
import com.elitesland.cbpl.scheduling.data.vo.param.ScheduleConfigSaveParamVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigDetailVO;
import com.elitesland.cbpl.scheduling.data.vo.resp.ScheduleConfigPagingVO;
import com.elitesland.cbpl.scheduling.registrar.DefaultSchedulingRegistrar;
import com.elitesland.cbpl.scheduling.registrar.domain.ScheduledTask;
import com.elitesland.cbpl.scheduling.util.SchedulingUtil;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.cron.CronPatternUtils;
import com.elitesland.cbpl.tool.db.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2023/09/06
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ScheduleConfigServiceImpl implements ScheduleConfigService {

    private final ScheduleConfigRepo scheduleConfigRepo;
    private final ScheduleConfigRepoProc scheduleConfigRepoProc;
    @Autowired(required = false)
    private DefaultSchedulingRegistrar schedulingRegistrar;
    private final RosefinchConfigService rosefinchConfigService;

    @Override
    public PagingVO<ScheduleConfigPagingVO> scheduleConfigPageBy(ScheduleConfigPagingParamVO query) {
        long count = scheduleConfigRepoProc.scheduleConfigCountBy(query);
        if (count > 0) {
            var list = scheduleConfigRepoProc.scheduleConfigPageBy(query);
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    @Override
    public List<ScheduledTask> scheduleConfigByParam(ScheduleConfigQueryParamVO query) {
        var tasks = scheduleConfigRepoProc.scheduleConfigByParam(query);
        return tasks.stream().map(SchedulingUtil::builder).collect(Collectors.toList());
    }

    @Override
    public ScheduleConfigDetailVO scheduleConfigById(Long id) {
        var schedule = scheduleConfigRepoProc.findById(id);
        if (ObjectUtil.isNull(schedule)) {
            throw PhoenixException.unchecked("Not Found Data");
        }
        return schedule;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(ScheduleConfigSaveParamVO saveParam) {
        ScheduledTask task = ScheduledTask.builder()
                .taskCode(saveParam.getTaskCode()).taskName(saveParam.getTaskName())
                .className(saveParam.getClassName()).methodName(saveParam.getMethod())
                .methodArgs(saveParam.getMethodArgs())
                .cron(saveParam.getCron())
                .build();
        boolean exists = scheduleConfigRepoProc.existsByCode(saveParam.getId(), saveParam.getTaskCode());
        Assert.isFalse(exists, "任务编码(" + saveParam.getTaskCode() + ")已存在");
        // 先 新增or更新 任务参数
        var rosefinchParam = ScheduleConfigConvert.INSTANCE.saveParamToRosefinch(saveParam);
        // 新增
        if (saveParam.isNew()) {
            // 先创建发布到任务中心
            Long rosefinchId = rosefinchConfigService.save(rosefinchParam);
            ScheduleConfigDO scheduleConfigDO = ScheduleConfigConvert.INSTANCE.saveParamToDO(saveParam);
            scheduleConfigDO.setRosefinchId(rosefinchId);
            scheduleConfigRepo.save(scheduleConfigDO);
            // 如果启用，加入定时任务
            if (rosefinchParam.getStatus().equals(ActiveStatus.ACTIVE.getCode())) {
                task.setRosefinchId(rosefinchId);
                addTriggerTask(task);
            }
            return scheduleConfigDO.getId();
        }
        // 修改
        else {
            Optional<ScheduleConfigDO> scheduleConfigDO = scheduleConfigRepo.findById(saveParam.getId());
            if (scheduleConfigDO.isEmpty()) {
                throw PhoenixException.unchecked("Not Found Data");
            }
            ScheduleConfigDO scheduleConfig = scheduleConfigDO.get();
            // 先更新任务中心参数
            rosefinchParam.setId(scheduleConfig.getRosefinchId());
            Long rosefinchId = rosefinchConfigService.save(rosefinchParam);
            // 再更新定时任务参数
            ScheduleConfigConvert.INSTANCE.saveParamMergeToDO(saveParam, scheduleConfig);
            scheduleConfig.setRosefinchId(rosefinchId);
            scheduleConfigRepo.save(scheduleConfig);
            // 如果启用，更新定时任务
            if (rosefinchParam.getStatus().equals(ActiveStatus.ACTIVE.getCode())) {
                task.setRosefinchId(rosefinchId);
                resetTriggerTask(task);
            }
            return scheduleConfig.getId();
        }
    }

    @Override
    public void active(Long id) {
        var schedule = scheduleConfigById(id);
        rosefinchConfigService.active(schedule.getRosefinchId());
        resetTriggerTask(scheduledByTaskId(id));
    }

    @Override
    public void inactive(Long id) {
        var schedule = scheduleConfigById(id);
        rosefinchConfigService.inactive(schedule.getRosefinchId());
        cancelTriggerTask(scheduledByTaskId(id));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<Long> ids) {
        List<Long> rosefinchIds = new ArrayList<>();
        // 取消定时任务
        ids.stream().map(id -> {
            var scheduledTask = scheduledByTaskId(id);
            rosefinchIds.add(scheduledTask.getRosefinchId());
            return scheduledTask;
        }).forEach(this::cancelTriggerTask);
        // 删除任务中心的关联数据
        rosefinchConfigService.delete(rosefinchIds);
        // 删除定时调度配置
        scheduleConfigRepo.deleteAllById(ids);
    }

    @Override
    public void execute(Long taskId) {
        try {
            // 读取任务配置
            runImmediately(scheduledByTaskId(taskId));
        } catch (RuntimeException e) {
            throw PhoenixException.unchecked("任务执行失败：" + e.getMessage());
        } catch (Exception e) {
            logger.error("[SCHEDULE] execute once error: {}", ExceptionUtils.formatException(e));
            throw PhoenixException.unchecked("任务执行异常");
        }
    }

    @Override
    public List<String> preview(Long taskId) {
        ScheduleConfigDetailVO schedule = scheduleConfigById(taskId);
        return CronPatternUtils.preview(schedule.getCron(), 20);
    }

    @Override
    public Set<String> taskCodes() {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            return schedulingRegistrar.taskCodes();
        }
        return Collections.emptySet();
    }

    @Override
    public boolean stopImmediately(String taskCode) {
        return this.stopImmediately(taskCode, true);
    }

    /**
     * 根据任务id，构建Schedule对象
     */
    private ScheduledTask scheduledByTaskId(Long taskId) {
        ScheduleConfigDetailVO schedule = scheduleConfigById(taskId);
        return SchedulingUtil.builder(schedule);
    }

    /**
     * 加入定时任务
     */
    private void addTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.addTriggerTask(task);
        }
    }

    /**
     * 重置定时任务
     */
    private void resetTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.resetTriggerTask(task);
        }
    }

    /**
     * 取消定时任务 - 执行中的任务不强制终止
     */
    private void cancelTriggerTask(ScheduledTask task) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            schedulingRegistrar.cancelTriggerTask(task);
        }
    }

    /**
     * 立即执行
     */
    private void runImmediately(ScheduledTask task) {
        if (ObjectUtil.isNull(schedulingRegistrar)) {
            throw new RuntimeException("Phoenix-Schedule模块未开启");
        }
        schedulingRegistrar.runImmediately(task);
    }

    /**
     * 取消/终止 任务
     *
     * @param mayInterruptIfRunning 是否立即终止当前任务
     * @return 如果无法取消任务，则为false，通常是因为任务已正常完成；否则为true；
     */
    private boolean stopImmediately(String taskCode, boolean mayInterruptIfRunning) {
        if (ObjectUtil.isNotNull(schedulingRegistrar)) {
            return schedulingRegistrar.stopImmediately(taskCode, mayInterruptIfRunning);
        }
        return true;
    }
}
