package com.elitesland.cbpl.scheduling.util;

import cn.hutool.extra.spring.SpringUtil;
import net.javacrumbs.shedlock.core.DefaultLockingTaskExecutor;
import net.javacrumbs.shedlock.core.LockConfiguration;
import net.javacrumbs.shedlock.core.LockProvider;
import net.javacrumbs.shedlock.core.LockingTaskExecutor;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.time.Instant;

import static com.elitesland.cbpl.scheduling.config.SchedulingProperties.*;

/**
 * 定时调度的分布式锁
 *
 * @author eric.hao
 * @since 2025/01/03
 */
public class SchedulingLock {

    private static LockProvider lockProvider;

    private static LockProvider getLockProvider() {
        if (lockProvider == null) {
            lockProvider = SpringUtil.getBean(LockProvider.class);
        }
        return lockProvider;
    }

    private static LockingTaskExecutor lockingTaskExecutor;

    private static LockingTaskExecutor getTaskExecutor() {
        if (lockingTaskExecutor == null) {
            lockingTaskExecutor = new DefaultLockingTaskExecutor(getLockProvider());
        }
        return lockingTaskExecutor;
    }

    /**
     * 加锁执行，定时任务场景
     *
     * @param taskKey  任务逐渐
     * @param executor 需要加锁的方法体
     * @since 0.2.2-SNAPSHOT
     */
    public static void schedulerWithLock(@NotBlank String taskKey, @NotNull Runnable executor) {
        // 持有锁的最长时间，防止宕机死锁
        Duration lockAtMostFor = Duration.ofSeconds(SCHEDULE_LOCK_AT_MOST_FOR);
        // 持有锁的最短时间，避免各个服务器的微小时钟差异，重复调用
        Duration lockAtLeastFor = Duration.ofSeconds(SCHEDULE_LOCK_AT_LEAST_FOR);
        getTaskExecutor().executeWithLock(executor,
                new LockConfiguration(Instant.now(), taskKey, lockAtMostFor, lockAtLeastFor));
    }
}
