package com.elitesland.cbpl.sns.bot.service;

import com.elitesland.cbpl.sns.bot.domain.BotMessage;
import com.elitesland.cbpl.sns.bot.domain.BotRequest;
import com.elitesland.cbpl.sns.bot.provider.BotMessageBuilder;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.core.exceptions.ExceptionUtils;
import com.github.jaemon.dinger.DingerSender;
import com.github.jaemon.dinger.core.entity.DingerRequest;
import com.github.jaemon.dinger.core.entity.enums.MessageSubType;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

/**
 * 告警机器人
 *
 * @author eric.hao
 * @since 2023/07/05
 */
@Slf4j
@RequiredArgsConstructor
public class BotSenderService {

    private static final int SEGMENT_LEN = 256;

    /**
     * 机器人消息推送
     */
    private final DingerSender dingerSender;

    /**
     * 消息模板构建
     */
    private final BotMessageBuilder messageBuilder;

    /**
     * 告警推送
     *
     * @param botSupplier 告警事件
     * @param message     告警信息
     */
    public void push(Supplier<BotRequest> botSupplier, String message) {
        try {
            // 参数构建
            BotMessage botMessage = new BotMessage();
            botMessage.setBizKey("");
            botMessage.setMessages(Collections.singletonList(message));
            BotRequest request = botSupplier.get();
            request.setMessages(Collections.singletonList(botMessage));
            // 消息推送
            push(request);
        } catch (Exception e) {
            logger.error("[PHOENIX-SNS] PUSH BOT ERR: {}", ExceptionUtils.formatException(e));
        }
    }

    /**
     * 告警推送
     *
     * @param botSupplier 告警事件
     * @param messages    告警信息
     */
    public void push(Supplier<BotRequest> botSupplier, List<BotMessage> messages) {
        try {
            // 参数构建
            BotRequest request = botSupplier.get();
            request.setMessages(messages);
            // 消息推送
            push(request);
        } catch (Exception e) {
            logger.error("[PHOENIX-SNS] PUSH BOT ERR: {}", ExceptionUtils.formatException(e));
        }
    }

    /**
     * 告警推送
     *
     * @param botSupplier 告警事件
     * @param e           异常堆栈
     */
    public void push(Supplier<BotRequest> botSupplier, Exception e) {
        push(botSupplier, ExceptionUtils.stacktraceToOneLineString(e, SEGMENT_LEN));
    }

    /**
     * 告警推送
     *
     * @param request 告警报文
     */
    public void push(BotRequest request) {
        DingerRequest msgReq = DingerRequest.request(messageBuilder.templateBuilder(request), request.getSnsTitle());
        logger.info("[PHOENIX-SNS] BOT REQUEST: {}", BeanUtils.toJsonStr(msgReq));
        dingerSender.send(MessageSubType.MARKDOWN, msgReq);
    }
}
