package com.elitesland.cbpl.sns.inbox.controller;

import com.elitesland.cbpl.sns.inbox.domain.InboxPayload;
import com.elitesland.cbpl.sns.inbox.domain.InboxType;
import com.elitesland.cbpl.sns.inbox.pipeline.InboxPipeline;
import com.elitesland.cbpl.sns.inbox.vo.param.InboxPageParamVO;
import com.elitesland.cbpl.tool.api.domain.HttpResult;
import com.elitesland.cbpl.tool.db.PagingVO;
import com.elitesland.cbpl.tool.iam.IamSpiUtil;
import com.elitesland.cbpl.unicom.util.UnicomClient;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.elitesland.cbpl.sns.inbox.config.SnsInboxProperties.SNS_INBOX_PIPELINE;

/**
 * 站内消息查询服务
 *
 * @author eric.hao
 * @since 2024/12/23
 */
@Api(tags = "消息数据服务")
@Slf4j
@RestController
@RequestMapping("/sns/inbox")
@RequiredArgsConstructor
public class InboxController {

    private final InboxPipeline inboxPipeline;

    @ApiOperation(value = "未读消息数量")
    @GetMapping("/count")
    public Map<InboxType, Long> countUnread() {
        return UnicomClient.supply(() ->
                inboxPipeline.countUnread(IamSpiUtil.currentUserIdStr(), InboxType.getStoreTypes()), SNS_INBOX_PIPELINE);
    }

    @ApiOperation(value = "分页查询")
    @PostMapping("/paging")
    public PagingVO<? extends InboxPayload> find(@RequestBody InboxPageParamVO query) {
        if (query.getRecipientId() == null) {
            query.setRecipientId(IamSpiUtil.currentUserIdStr());
        }
        return UnicomClient.supply(() -> inboxPipeline.find(query), SNS_INBOX_PIPELINE);
    }

    @ApiOperation(value = "明细查询")
    @PatchMapping("/{type}/messages/{id}")
    public Optional<? extends InboxPayload> read(
            @PathVariable InboxType type, @PathVariable("id") String messageId) {
        return UnicomClient.supply(() -> inboxPipeline.read(type, messageId, IamSpiUtil.currentUserIdStr()), SNS_INBOX_PIPELINE);
    }

    @ApiOperation(value = "全部已读")
    @PatchMapping("/{type}/allMessages")
    public HttpResult<Void> readAllMessages(@PathVariable InboxType type) {
        UnicomClient.run(() -> inboxPipeline.readAllMessages(type, IamSpiUtil.currentUserIdStr()), SNS_INBOX_PIPELINE);
        return HttpResult.ok();
    }

    @ApiOperation(value = "消息类型下拉")
    @PostMapping("/msgType")
    public HttpResult<List<Map<String, String>>> getMsgType() {
        return HttpResult.ok(UnicomClient.supply(inboxPipeline::getMsgType, SNS_INBOX_PIPELINE));
    }
}
