package com.elitesland.cbpl.sns.inbox.pipeline.mongo.service;

import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.map.MapUtil;
import com.elitesland.cbpl.sns.inbox.domain.InboxPayload;
import com.elitesland.cbpl.sns.inbox.domain.InboxType;
import com.elitesland.cbpl.sns.inbox.pipeline.InboxPipeline;
import com.elitesland.cbpl.sns.inbox.pipeline.mongo.query.InboxPageQuery;
import com.elitesland.cbpl.sns.inbox.pipeline.mongo.query.InboxUnreadQuery;
import com.elitesland.cbpl.sns.inbox.vo.param.InboxPageParamVO;
import com.elitesland.cbpl.sns.inbox.vo.resp.InboxRespVO;
import com.elitesland.cbpl.tool.db.PagingVO;
import com.elitesland.cbpl.tool.mongo.repository.MongoStorer;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.springframework.data.mongodb.core.query.Criteria.where;

/**
 * @author eric.hao
 * @since 2024/12/19
 */
@Slf4j
@Service
@RequiredArgsConstructor
@UnicomTag("MONGO")
public class InboxMongoService implements InboxPipeline {

    private static final String ERR_MONGO_CONFIGURATION = "[SNS-INBOX] MongoDB Incorrect configuration.";

    @Autowired(required = false)
    private MongoStorer mongoStorer;
    @Autowired(required = false)
    private MongoTemplate mongoTemplate;

    private Class<InboxRespVO> getEntityClass() {
        return InboxRespVO.class;
    }

    @Override
    public void save(InboxPayload payload) {
        if (mongoStorer == null) {
            logger.error(ERR_MONGO_CONFIGURATION);
            return;
        }
        try {
            mongoStorer.save(payload);
        } catch (Throwable throwable) {
            logger.error("[SNS-INBOX] persistence mongodb error", throwable);
        }
    }

    @Override
    public Map<InboxType, Long> countUnread(String reader, Stream<InboxType> types) {
        if (mongoStorer == null) {
            logger.error(ERR_MONGO_CONFIGURATION);
            return MapUtil.empty();
        }

        return types.collect(Collectors.toMap(
                type -> type,
                type -> InboxUnreadQuery.builder()
                        .type(type).reader(reader)
                        .build().count(mongoTemplate)
        ));
    }

    @Override
    public PagingVO<? extends InboxPayload> find(InboxPageParamVO params) {
        if (mongoStorer == null) {
            logger.error(ERR_MONGO_CONFIGURATION);
            return new PagingVO<>();
        }
        // 分页查询
        PagingVO<InboxRespVO> result = InboxPageQuery.builder().params(params).build()
                .paging(mongoTemplate, getEntityClass());

        // 处理已读未读，存在一个消息多个收件人，某个收件人已读问题
        if (result.getTotal() > 0) {
            List<InboxRespVO> records = result.getRecords();
//            Map<String, List<MsgTemplateDO>> tmplList = msgTemplateRepo.findAll().stream().collect(Collectors.groupingBy(MsgTemplateDO::getTmplCode));
//            Map<String, List<MsgTemplateLinkDO>> templateLinkList = msgTemplateLinkRepo.findAll().stream().collect(Collectors.groupingBy(MsgTemplateLinkDO::getLinkCode));
            records.forEach(record -> {
                // TODO 对站内信的跳转参数值做处理，保存在extInfo中
//                Map<String, Object> extInfo = BeanUtils.beanToMap(record.getExtInfo());
//                if (extInfo.get("eventType") != null) {
//                    extInfo.put("msgTypeName", CollectionUtils.isNotEmpty(tmplList.get(extInfo.get("eventType").toString()))
//                            ? tmplList.get(extInfo.get("eventType").toString()).get(0).getTmplName() : "");
//                    String tempCode = CollectionUtils.isNotEmpty(tmplList.get(extInfo.get("eventType").toString())) ?
//                            tmplList.get(extInfo.get("eventType").toString()).get(0).getTmplDesc() : "";
//                    if (StringUtils.isNotEmpty(tempCode)) {
//                        extInfo.put("linkUrl2", CollectionUtils.isNotEmpty(templateLinkList.get(tempCode)) ?
//                                templateLinkList.get(tempCode).get(0).getLinkVal2() : "");
//                    }
//                }
//                if (extInfo.get("linkUrl") == null && StringUtils.isNotEmpty(record.getContent())) {
//                    String linkUrl = inboxUtil.getTempLinkUrl(record.getContent());
//                    extInfo.put("linkUrl", linkUrl);
//                }
//                record.setExtInfo(extInfo);

                record.setReadFlag(!record.getRead().isEmpty() && record.getRead().containsKey(params.getRecipientId()));
            });
        }
        return result;
    }

    @Override
    public Optional<? extends InboxPayload> read(InboxType type, String messageId, String reader) {
        if (mongoStorer == null) {
            logger.error(ERR_MONGO_CONFIGURATION);
            return Optional.empty();
        }

        var message = mongoTemplate.findById(messageId, getEntityClass(), type.getStoreName());
        if (message == null) {
            return Optional.empty();
        }

        var update = Update.update("read." + reader, Instant.now());
        var query = Query.query(
                where("_id").is(messageId).orOperator(
                        where("tos").size(0),
                        where("tos").is(reader)));
        mongoTemplate.findAndModify(query, update, getEntityClass(), type.getStoreName());
        return Optional.of(message);
    }

    @Override
    public void readAllMessages(InboxType type, String reader) {
        if (mongoStorer == null) {
            logger.error(ERR_MONGO_CONFIGURATION);
            return;
        }
        var update = Update.update("read." + reader, Instant.now());
        var query = Query.query(new Criteria().orOperator(
                where("tos").size(0),
                where("tos").is(reader)));
        mongoTemplate.updateMulti(query, update, getEntityClass(), type.getStoreName());
    }

    @Override
    public List<Map<String, String>> getMsgType() {
        if (mongoStorer == null) {
            logger.error(ERR_MONGO_CONFIGURATION);
            return ListUtil.empty();
        }

        // TODO 消息类型下拉，从配置的消息模版中获取
//        List<MsgTemplateDO> tmplDo = msgTemplateRepo.findAll();
//        List<Map<String, String>> result = new ArrayList<>();
//        tmplDo.forEach(tmpl -> {
//            Map<String, String> map = new HashMap<>();
//            map.put("code", tmpl.getTmplCode());
//            map.put("name", tmpl.getTmplName());
//            result.add(map);
//        });
//        return result;
        return ListUtil.empty();
    }
}
