package com.elitesland.cbpl.sns.inbox.service;

import com.elitesland.cbpl.sns.common.domain.SnsPayload;
import com.elitesland.cbpl.sns.inbox.domain.InboxPayload;
import com.elitesland.cbpl.sns.inbox.domain.InboxType;
import com.elitesland.cbpl.sns.inbox.pipeline.InboxPipeline;
import com.elitesland.cbpl.sns.inbox.vo.save.InboxSaveParamVO;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import com.elitesland.cbpl.tool.websocket.notifier.WebSocketNotifier;
import com.elitesland.cbpl.unicom.util.UnicomClient;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Set;

import static com.elitesland.cbpl.sns.inbox.config.SnsInboxProperties.SNS_INBOX_PIPELINE;

/**
 * @author eric.hao
 * @since 2024/12/19
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class InboxServiceImpl implements InboxService {

    private final InboxPipeline inboxPipeline;

    @Override
    public void sendInbox(InboxType inboxType, Set<String> tos, SnsPayload payload) {
        if (!payload.isValid()) {
            logger.info("[SNS-INBOX] INBOX message payload is empty.");
            return;
        }
        try {
            logger.info("[SNS-INBOX] INBOX message to userId={} sending...", tos);
            var inboxPayload = InboxSaveParamVO.builder()
                    .createTime(LocalDateTime.now())
                    .type(inboxType)
                    .subject(payload.getSubject()).content(payload.getContent())
                    .from("1").tos(tos)
//                    .extInfo(payload.getExtInfo())
                    .build();
            logger.info("[SNS-INBOX] INBOX payload: {}", inboxPayload);
            this.push(inboxPayload);
            logger.info("[SNS-INBOX] INBOX message SENT.");
        } catch (Throwable e) {
            logger.error("[SNS-INBOX] INBOX message sent FAILED.", e);
        }
    }

    /**
     * 站内信消息持久化，并推送给客户端
     */
    private void push(InboxPayload payload) {
        // 1.保存
        UnicomClient.run(() -> inboxPipeline.save(payload), SNS_INBOX_PIPELINE);
        // 2.推送
        if (SpringUtils.isPresent(WebSocketNotifier.class)) {
            var notifier = SpringUtils.getBean(WebSocketNotifier.class);
            notifier.notify(payload);
        } else {
            logger.warn("[SNS-INBOX] WebSocketNotifier closed.");
        }
    }
}
