package com.elitesland.cbpl.unicom.config;

import com.elitesland.cbpl.unicom.adapter.DefaultUnicomAdapter;
import com.elitesland.cbpl.unicom.proxy.UnicomAdapterProxy;
import com.elitesland.cbpl.unicom.proxy.UnicomAdapterProxyFactoryBean;
import com.elitesland.cbpl.unicom.proxy.UnicomInterfaceProxy;
import com.elitesland.cbpl.unicom.proxy.UnicomInterfaceProxyFactoryBean;
import com.elitesland.cbpl.unicom.register.ProxyBeanDefinitionRegister;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;

import static com.elitesland.cbpl.unicom.config.UnicomProperties.UNICOM_PREFIX;
import static org.reflections.scanners.Scanners.SubTypes;
import static org.reflections.scanners.Scanners.TypesAnnotated;

/**
 * @author eric.hao
 * @since 2022/06/29
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(UnicomProperties.class)
@ConditionalOnProperty(prefix = UNICOM_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class UnicomConfig implements EnvironmentAware {

    private UnicomProperties properties;

    /**
     * 反射类扫描注册
     */
    @Bean
    @ConditionalOnMissingBean
    public Reflections reflections() {
        logger.info("[PHOENIX-UNICOM] loading reflections.");
        Collection<URL> urls = new ArrayList<>();
        for (String basePackage : properties.getBasePackage()) {
            urls.addAll(ClasspathHelper.forPackage(basePackage));
        }
        return new Reflections(new ConfigurationBuilder().setUrls(urls).addScanners(TypesAnnotated, SubTypes));
    }

    /**
     * 注册代理类
     */
    @Bean
    public ProxyBeanDefinitionRegister proxyBeanDefinitionRegister(Reflections reflections) {
        logger.info("[PHOENIX-UNICOM] load unicom proxy register.");
        return new ProxyBeanDefinitionRegister(reflections, properties);
    }

    /**
     * 接口拦截代理
     */
    @Bean
    public UnicomInterfaceProxy unicomInterfaceProxy(Reflections reflections) {
        logger.info("[PHOENIX-UNICOM] load unicom interface proxy.");
        return new UnicomInterfaceProxy(reflections, properties);
    }

    @Bean
    public UnicomInterfaceProxyFactoryBean unicomInterfaceProxyFactoryBean() {
        return new UnicomInterfaceProxyFactoryBean();
    }

    @Bean
    public UnicomAdapterProxy unicomAdapterProxy(Reflections reflections) {
        logger.info("[PHOENIX-UNICOM] load unicom adapter proxy.");
        return new UnicomAdapterProxy(reflections, properties);
    }

    @Bean
    public UnicomAdapterProxyFactoryBean unicomAdapterProxyFactoryBean() {
        return new UnicomAdapterProxyFactoryBean();
    }


    @Bean
    public DefaultUnicomAdapter defaultUnicomAdapter() {
        return new DefaultUnicomAdapter();
    }

    @Override
    public void setEnvironment(Environment environment) {
        this.properties = UnicomProperties.from(environment);
    }
}
