package com.elitesland.cbpl.unicom.config;

import cn.hutool.core.collection.CollUtil;
import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.core.env.Environment;

import java.util.List;
import java.util.stream.Collectors;

/**
 * Unicom 配置类
 *
 * @author eric.hao
 * @since 2022/09/14
 */
@Data
@ConfigurationProperties(prefix = UnicomProperties.UNICOM_PREFIX)
public class UnicomProperties {

    public static final String UNICOM_PREFIX = "cbpl.unicom";

    /**
     * 是否启用：默认关闭
     */
    private boolean enabled;

    /**
     * 扫描路径
     */
    private List<String> basePackage = CollUtil.toList("com.elitesland", "com.elitescloud");

    /**
     * 业务域 - 调用方式
     */
    private List<InvokeModeConfig> businessDomain;

    /**
     * 默认执行的策略
     */
    private String defaultAdapter;

    /**
     * 根据领域，匹配调用方式
     *
     * @param domainName 领域
     * @return 调用方式
     */
    public InvokeModeConfig invokeMode(String domainName) {
        if (CollUtil.isEmpty(businessDomain)) {
            return null;
        }
        List<InvokeModeConfig> collect = businessDomain.stream().filter(row -> row.getName().equals(domainName)).collect(Collectors.toList());
        if (CollUtil.isEmpty(collect)) {
            return null;
        }
        if (collect.size() > 1) {
            throw new IllegalArgumentException("[UNICOM-CONFIG] cbpl.unicom.business-domain, name[" + domainName + "] multiple configurations.");
        }
        return collect.get(0);
    }

    /**
     * 读取环境配置
     */
    public static UnicomProperties from(Environment environment) {
        Binder env = Binder.get(environment);
        BindResult<UnicomProperties> properties = env.bind(UNICOM_PREFIX, UnicomProperties.class);
        return properties.isBound() ? properties.get() : new UnicomProperties();
    }
}
