package com.elitesland.cbpl.unicom.proxy;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.spring.SpringUtil;
import com.elitesland.cbpl.unicom.adapter.DefaultUnicomAdapter;
import com.elitesland.cbpl.unicom.adapter.SpecifyAdapter;
import com.elitesland.cbpl.unicom.adapter.UnicomAdapter;
import com.elitesland.cbpl.unicom.adapter.domain.UnicomSession;
import com.elitesland.cbpl.unicom.annotation.Unicom;
import com.elitesland.cbpl.unicom.config.UnicomProperties;
import com.elitesland.cbpl.unicom.util.ParamAnalyser;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.springframework.cglib.proxy.MethodInterceptor;
import org.springframework.cglib.proxy.MethodProxy;

import java.lang.reflect.Method;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2022/06/28
 */
@Slf4j
@RequiredArgsConstructor
public class UnicomInterfaceProxy implements MethodInterceptor {

    private final Reflections reflections;
    private final UnicomProperties properties;

    /**
     * 委托方法被调用时转发
     *
     * @param obj         代理子类实例
     * @param method      委托类方法反射
     * @param args        方法参数对象
     * @param methodProxy 用于调用原方法
     */
    @Override
    public Object intercept(Object obj, Method method, Object[] args, MethodProxy methodProxy) throws Throwable {
        Class<?> superClass = method.getDeclaringClass();
        Unicom anno = superClass.getAnnotation(Unicom.class);
        List<Object> providers = reflections.getSubTypesOf(superClass).stream().map(SpringUtil::getBean).collect(Collectors.toList());
        if (CollUtil.isEmpty(providers)) {
            if (anno.throwException()) {
                throw new RuntimeException("[PHOENIX-UNICOM] " + superClass + "：未找到任何实现类");
            } else {
                logger.info("[PHOENIX-UNICOM] {}：未找到任何实现类", superClass);
                return null;
            }
        }
        Object provider = loadAdapter(superClass).filter(providers);
        return methodProxy.invoke(provider, ParamAnalyser.parse(provider, method, args));
    }

    /**
     * 代理适配实现类
     */
    private UnicomAdapter loadAdapter(Class<?> superClass) {
        // 1. 优先使用上下文的标签值
        String unicomTag = UnicomSession.getCurrentUnicomTag();
        if(StrUtil.isNotBlank(unicomTag)) {
            logger.debug("[PHOENIX-UNICOM] use adapter: com.elitesland.cbpl.unicom.adapter.SpecifyAdapter");
            return SpringUtil.getBean(SpecifyAdapter.class);
        }
        // 2. 其次，优先使用@Unicom的指定策略
        Unicom anno = superClass.getAnnotation(Unicom.class);
        Class<?> adapterClass = anno.adapter();
        // 3. 再次，使用读取配置文件中，指定默认执行的策略
        if (adapterClass.equals(DefaultUnicomAdapter.class) && StrUtil.isNotEmpty(properties.getDefaultAdapter())) {
            String defaultAdapter = properties.getDefaultAdapter();
            logger.debug("[PHOENIX-UNICOM] use adapter: {}", defaultAdapter);
            return SpringUtil.getBean(defaultAdapter);
        }
        logger.debug("[PHOENIX-UNICOM] use adapter: {}", adapterClass.getName());
        // 4. 以上都不满足时，会返回使用`DefaultUnicomAdapter.class`
        return (UnicomAdapter) SpringUtil.getBean(adapterClass);
    }
}