package com.elitesland.cbpl.unionpay.shoupay.data.service.impl;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.unionpay.shoupay.cache.ShouPayCache;
import com.elitesland.cbpl.unionpay.shoupay.cache.ShouPayDeviceVO;
import com.elitesland.cbpl.unionpay.shoupay.data.convert.ShouPayDeviceConvert;
import com.elitesland.cbpl.unionpay.shoupay.data.vo.param.ShouPayDeviceSaveParamVO;
import com.elitesland.cbpl.unionpay.shoupay.data.entity.ShouPayDeviceDO;
import com.elitesland.cbpl.unionpay.shoupay.data.repo.ShouPayDeviceRepo;
import com.elitesland.cbpl.unionpay.shoupay.data.repo.ShouPayDeviceRepoProc;
import com.elitesland.cbpl.unionpay.shoupay.data.service.ShouPayDeviceService;
import com.elitesland.cbpl.unionpay.shoupay.data.vo.param.ShouPayDeviceQueryParamVO;
import com.elitesland.cbpl.unionpay.shoupay.data.vo.resp.ShouPayDeviceDetailVO;
import com.elitesland.cbpl.unionpay.shoupay.data.vo.resp.ShouPayDeviceRespVO;
import com.elitesland.cbpl.unionpay.shoupay.domain.base.ShouPayRespVO;
import com.elitesland.cbpl.unionpay.shoupay.domain.resp.ShouPayTerminalRespVO;
import com.elitesland.cbpl.unionpay.shoupay.proxy.ShouPayProxy;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

import static com.elitesland.cbpl.unionpay.shoupay.config.ShouPayProperties.SHOU_PAY_ENABLED;

/**
 * @author eric.hao
 * @since 2023/11/27
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ShouPayDeviceServiceImpl implements ShouPayDeviceService {

    private final ShouPayDeviceRepo shouPayDeviceRepo;
    private final ShouPayDeviceRepoProc shouPayDeviceRepoProc;

    @Override
    public List<ShouPayDeviceRespVO> shouPayDeviceByParam(ShouPayDeviceQueryParamVO query) {
        return shouPayDeviceRepoProc.shouPayDeviceByParam(query);
    }

    @Override
    public ShouPayDeviceDetailVO shouPayDeviceById(String deviceId) {
        Optional<ShouPayDeviceDO> shouPayDeviceDO = shouPayDeviceRepo.findByDeviceId(deviceId);
        if (!shouPayDeviceDO.isPresent()) {
            throw new RuntimeException("Not Found Data");
        }
        return ShouPayDeviceConvert.INSTANCE.doToVO(shouPayDeviceDO.get());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Long save(ShouPayDeviceSaveParamVO saveParam) {
        // 新增
        if (saveParam.isNew()) {
            ShouPayDeviceDO shouPayDeviceDO = ShouPayDeviceConvert.INSTANCE.saveParamToDO(saveParam);
            shouPayDeviceRepo.save(shouPayDeviceDO);
            return shouPayDeviceDO.getId();
        }
        // 修改
        else {
            Optional<ShouPayDeviceDO> shouPayDeviceDO = shouPayDeviceRepo.findById(saveParam.getId());
            if (!shouPayDeviceDO.isPresent()) {
                throw new RuntimeException("Not Found Data");
            }
            ShouPayDeviceDO shouPayDevice = shouPayDeviceDO.get();
            ShouPayDeviceConvert.INSTANCE.saveParamMergeToDO(saveParam, shouPayDevice);
            shouPayDeviceRepo.save(shouPayDevice);
            return shouPayDevice.getId();
        }
    }

    @Autowired(required = false)
    private ShouPayProxy shouPayProxy;
    @Autowired(required = false)
    private ShouPayCache shouPayCache;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ShouPayTerminalRespVO activate(String deviceId) {
        if (!SHOU_PAY_ENABLED) {
            throw new RuntimeException("收钱吧配置未开启");
        }
        // 设备信息
        ShouPayDeviceVO device = shouPayCache.get(deviceId);
        if (StrUtil.isNotBlank(device.getTerminalSn())) {
            throw new RuntimeException("该设备已经激活，不用重复操作");
        }
        ShouPayRespVO<ShouPayTerminalRespVO> result = shouPayProxy.activate(deviceId);
        if (result.fail()) {
            throw new RuntimeException("该设备激活失败");
        }
        if (!result.getData().valid()) {
            throw new RuntimeException("该设备激活错误");
        }
        logger.debug("[PHOENIX-UNIONPAY][SHOUPAY-DEBUG] activate result: {}", BeanUtils.toJsonStr(result));
        shouPayDeviceRepoProc.updateTerminal(deviceId, result.getData());
        return result.getData();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ShouPayTerminalRespVO checkin(String deviceId) {
        if (!SHOU_PAY_ENABLED) {
            throw new RuntimeException("收钱吧配置未开启");
        }
        ShouPayRespVO<ShouPayTerminalRespVO> result = shouPayProxy.checkin(deviceId);
        if (result.fail()) {
            throw new RuntimeException("该设备签到失败");
        }
        if (!result.getData().valid()) {
            throw new RuntimeException("该设备签到错误");
        }
        logger.debug("[PHOENIX-UNIONPAY][SHOUPAY-DEBUG] checkin result: {}", BeanUtils.toJsonStr(result));
        shouPayDeviceRepoProc.updateTerminal(deviceId, result.getData());
        return result.getData();
    }
}
