package com.elitesland.cbpl.unionpay.shoupay.proxy;

import cn.hutool.system.SystemUtil;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.unionpay.shoupay.cache.ShouPayCache;
import com.elitesland.cbpl.unionpay.shoupay.cache.ShouPayDeviceVO;
import com.elitesland.cbpl.unionpay.shoupay.common.util.HttpUtil;
import com.elitesland.cbpl.unionpay.shoupay.domain.base.ShouPayRespVO;
import com.elitesland.cbpl.unionpay.shoupay.domain.param.*;
import com.elitesland.cbpl.unionpay.shoupay.domain.resp.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.validation.annotation.Validated;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.HashMap;
import java.util.Map;

import static com.elitesland.cbpl.unionpay.shoupay.common.constant.ShouPayConstant.*;

/**
 * 根据收钱吧提供的示例，改造封装的代理工具类
 *
 * @author eric.hao
 * @since 2023/11/27
 */
@Slf4j
@Validated
@Component
@RequiredArgsConstructor
public class ShouPayProxy {

    private final ShouPayCache shouPayCache;

    /**
     * 终端激活
     *
     * @param deviceId 收银终端序列号【建议格式为`品牌名+门店编号+'POS'+POS编号`】
     * @return {terminal_sn:"$终端号", terminal_key:"$终端密钥"}
     */
    public ShouPayRespVO<ShouPayTerminalRespVO> activate(@NotBlank String deviceId) {
        // 设备信息
        ShouPayDeviceVO device = shouPayCache.get(deviceId);

        Map<String, Object> params = new HashMap<>();
        // app_id
        params.put("app_id", device.getAppId());
        // 激活码
        params.put("code", device.getActivateCode());
        // 收银终端序列号
        params.put("device_id", deviceId);
        // 收银终端编号
        params.put("client_sn", device.getClientSn());
        // 终端名称
        params.put("name", device.getDeviceName());
        // 当前系统信息
        params.put("os_info", SystemUtil.getOsInfo().getName());
        // SDK版本
        params.put("sdk_version", device.getSdkVersion());

        // 发送请求
        String url = device.getApiDomain() + TERMINAL_ACTIVATE;
        return HttpUtil.httpPost(url, params, device.getVendorKey(), device.getVendorSn(), ShouPayTerminalRespVO.class);
    }

    /**
     * 终端签到
     *
     * @param deviceId – 收银终端序列号
     * @return {terminal_sn:"$终端号", terminal_key:"$终端密钥"}
     */
    public ShouPayRespVO<ShouPayTerminalRespVO> checkin(@NotBlank String deviceId) {
        // 设备信息
        ShouPayDeviceVO device = shouPayCache.get(deviceId);
        // 发送请求
        return terminalPost(deviceId, TERMINAL_CHECK_IN, ShouPayTerminalParamVO.checkin(device), ShouPayTerminalRespVO.class);
    }

    /**
     * 检查终端密钥
     *
     * @param deviceId 设备号
     */
    private void checkTerminalKey(String deviceId) {
        // 设备信息
        ShouPayDeviceVO device = shouPayCache.get(deviceId);
        // 需要重新签到
        if (device.needCheckin()) {
            // 重新签到
            checkin(deviceId);
        }
    }

    /**
     * 付款
     *
     * @param deviceId – 收银终端序列号
     * @return 交易结果
     */
    public ShouPayRespVO<ShouPayTradeRespVO> pay(@NotBlank String deviceId, @Valid ShouPayTradeParamVO paramVO) {
        return terminalCheckPost(deviceId, UPAY_V2_PAY, paramVO, ShouPayTradeRespVO.class);
    }

    /**
     * 退款
     *
     * @param deviceId – 收银终端序列号
     * @return 退款结果
     */
    public ShouPayRespVO<ShouPayRefundRespVO> refund(@NotBlank String deviceId, @Valid ShouPayRefundParamVO paramVO) {
        return terminalCheckPost(deviceId, UPAY_V2_REFUND, paramVO, ShouPayRefundRespVO.class);
    }

    /**
     * 查询
     *
     * @param deviceId – 收银终端序列号
     * @return 查询结果订单
     */
    public ShouPayRespVO<ShouPayQueryRespVO> query(@NotBlank String deviceId, @Valid ShouPayQueryParamVO paramVO) {
        return terminalCheckPost(deviceId, UPAY_V2_QUERY, paramVO, ShouPayQueryRespVO.class);
    }

    /**
     * 自动撤单
     *
     * @param deviceId – 收银终端序列号
     * @return 撤单结果
     */
    public ShouPayRespVO<ShouPayCancelRespVO> cancel(@NotBlank String deviceId, @Valid ShouPayCancelParamVO paramVO) {
        return terminalCheckPost(deviceId, UPAY_V2_CANCEL, paramVO, ShouPayCancelRespVO.class);
    }

    /**
     * 手动撤单
     *
     * @param deviceId – 收银终端序列号
     * @return 撤单结果
     */
    public ShouPayRespVO<ShouPayCancelRespVO> revoke(@NotBlank String deviceId, @Valid ShouPayCancelParamVO paramVO) {
        return terminalCheckPost(deviceId, UPAY_V2_REVOKE, paramVO, ShouPayCancelRespVO.class);
    }

    /**
     * 设备业务接口请求
     *
     * @param deviceId 设备号
     * @param url      请求地址
     * @param paramVO  请求参数
     * @param clazz    返回对象
     * @return 响应字符串结果
     */
    private <T> ShouPayRespVO<T> terminalCheckPost(String deviceId, String url, Object paramVO, Class<T> clazz) {
        // 检查并更新终端密钥
        checkTerminalKey(deviceId);
        return terminalPost(deviceId, url, paramVO, clazz);
    }

    /**
     * 设备业务接口请求
     *
     * @param deviceId 设备号
     * @param url      请求地址
     * @param paramVO  请求参数
     * @param clazz    返回对象
     * @return 响应字符串结果
     */
    private <T> ShouPayRespVO<T> terminalPost(String deviceId, String url, Object paramVO, Class<T> clazz) {
        // 设备信息
        ShouPayDeviceVO device = shouPayCache.get(deviceId);
        // 业务参数
        Map<String, Object> params = HttpUtil.paramToMap(paramVO);
        params.put("terminal_sn", device.getTerminalSn());
        // 接口请求
        ShouPayRespVO<T> result = HttpUtil.httpPost(device.getApiDomain() + url, params, device.getTerminalKey(), device.getTerminalSn(), clazz);
        // 请求失败
        if (result.fail()) {
            logger.error("[PHOENIX-UNIONPAY][SHOUPAY-ERROR] request failed: deviceId({}), url({}).", deviceId, url);
        }
        logger.debug("[PHOENIX-UNIONPAY][SHOUPAY-DEBUG] request url({}), result: {}", url, BeanUtils.toJsonStr(result));
        return result;
    }
}
