package com.elitesland.cbpl.rosefinch.service.impl;

import com.elitesland.cbpl.rosefinch.convert.RosefinchConvert;
import com.elitesland.cbpl.rosefinch.entity.RosefinchDetailDO;
import com.elitesland.cbpl.rosefinch.entity.RosefinchLogDO;
import com.elitesland.cbpl.rosefinch.repo.RosefinchDetailRepo;
import com.elitesland.cbpl.rosefinch.repo.RosefinchDetailRepoProc;
import com.elitesland.cbpl.rosefinch.repo.RosefinchRepo;
import com.elitesland.cbpl.rosefinch.repo.RosefinchRepoProc;
import com.elitesland.cbpl.rosefinch.service.RosefinchService;
import com.elitesland.cbpl.rosefinch.udc.LogStatusEnum;
import com.elitesland.cbpl.rosefinch.vo.param.RosefinchPageParamVO;
import com.elitesland.cbpl.rosefinch.vo.resp.RosefinchListRespVO;
import com.elitesland.cbpl.rosefinch.vo.resp.RosefinchRespVO;
import com.elitesland.yst.common.base.ApiCode;
import com.elitesland.yst.common.base.PagingVO;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.system.service.SysNumberRuleService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboReference;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * @author eric.hao
 * @since 2022/06/20
 */
@Slf4j
@Service
@AllArgsConstructor
public class RosefinchServiceImpl implements RosefinchService {

    private final RosefinchRepo rosefinchRepo;
    private final RosefinchRepoProc rosefinchRepoProc;
    private final RosefinchDetailRepo rosefinchDetailRepo;
    private final RosefinchDetailRepoProc rosefinchDetailRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public RosefinchLogDO init(String asyncType, int totalCount) {
        RosefinchLogDO headLog = new RosefinchLogDO();
        String logNo = generateCode("ASYNC_TASK", new ArrayList<>());
        headLog.setLogNo(logNo);
        headLog.setLogStatus(LogStatusEnum.FALSE.getValue());
        headLog.setLogType(asyncType);
        headLog.setSuccessCount(0);
        headLog.setFailCount(0);
        headLog.setTotalCount(totalCount);
        rosefinchRepo.save(headLog);
        return headLog;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void save(RosefinchLogDO headLog, List<RosefinchDetailDO> detailLogList) {
        // 导入日志
        rosefinchRepo.save(headLog);
        // 行明细日志
        detailLogList.forEach(row -> {
            row.setMasId(headLog.getId());
            int idx = Math.min(row.getErrorMsg().length(), 255);
            row.setErrorMsg(row.getErrorMsg().substring(0, idx));
            rosefinchDetailRepo.save(row);
        });
    }

    @Override
    public PagingVO<RosefinchListRespVO> rosefinchPageBy(RosefinchPageParamVO paramVO) {
        long count = rosefinchRepoProc.rosefinchCountBy(paramVO);
        if (count > 0) {
            var list = rosefinchRepoProc.rosefinchPageBy(paramVO);
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    @Override
    public RosefinchRespVO rosefinchById(long logId) {
        Optional<RosefinchLogDO> rosefinchDO = rosefinchRepo.findById(logId);
        if (rosefinchDO.isEmpty()) {
            throw new BusinessException("[批处理日志] " + logId + " 不存在");
        }
        return RosefinchConvert.INSTANCE.doToVO(rosefinchDO.get());
    }

    @DubboReference(version = "${provider.service.version}")
    private SysNumberRuleService sysNumberRuleService;

    private String generateCode(String ruleCode, List<String> runtimeValues) {
        try {
            val code = sysNumberRuleService.generateCode(ruleCode, runtimeValues);
            if (StringUtils.isEmpty(code)) {
                logger.error("调用发号器异常,发号器编码未生成,规则:" + ruleCode);
                throw new BusinessException(ApiCode.FAIL, "调用发号器异常,发号器编码未生成,规则:" + ruleCode);
            }
            return code;
        } catch (Exception e) {
            logger.error("generateCode error:", e);
            throw new BusinessException(ApiCode.FAIL, e.getMessage());
        }
    }
}
