package com.elitesland.cbpl.unicom.util;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.iam.IamUserUtil;
import com.elitesland.cbpl.unicom.adapter.UnicomAdapter;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import com.elitesland.cbpl.unicom.context.Customize;
import com.elitesland.cbpl.unicom.context.Industry;
import com.elitesland.cbpl.unicom.context.ProcContext;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.system.dto.SysTenantDTO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * @author eric.hao
 * @since 2022/03/21
 */
@Slf4j
@Component
public class TenantAdapter implements UnicomAdapter {

    private static final String ERR_TENANT_CONTEXT_CONFIG = "找不到租户的实施方案，请联系管理员";

    /**
     * 根据当前登录用户，找到租户对应的解决方案
     */
    @Override
    public Object filter(List<Object> providers) {
        // 1.先找定制方案
        var customerProvider = providers.stream().filter(this::matchCustomize).findAny();
        if (customerProvider.isPresent()) {
            logger.debug("[CBPL-PHOENIX] context provider: {}", customerProvider.getClass().getSimpleName());
            return customerProvider.get();
        }

        // 2.再找行业方案
        var industryProvider = providers.stream().filter(this::matchIndustry).findAny();
        if (industryProvider.isPresent()) {
            logger.debug("[CBPL-PHOENIX] context provider: {}", industryProvider.getClass().getSimpleName());
            return industryProvider.get();
        }

        // 3.最后执行标准方案
        var generalProvider = providers.stream().filter(this::matchGeneral).findAny()
                .orElseThrow(() -> {
                    throw new BusinessException(ERR_TENANT_CONTEXT_CONFIG);
                });
        logger.debug("[CBPL-PHOENIX] context provider: {}", generalProvider.getClass().getSimpleName());
        return generalProvider;
    }

    private boolean matchCustomize(Object provider) {
        UnicomTag annotation = provider.getClass().getAnnotation(UnicomTag.class);
        return currentCustomize().match(Customize.fromCode(annotation.value()));
    }

    private boolean matchIndustry(Object provider) {
        UnicomTag annotation = provider.getClass().getAnnotation(UnicomTag.class);
        return currentIndustry().match(Industry.fromCode(annotation.value()));
    }

    private boolean matchGeneral(Object provider) {
        UnicomTag annotation = provider.getClass().getAnnotation(UnicomTag.class);
        return general().match(Industry.fromCode(annotation.value()));
    }

    /**
     * 标准实施
     */
    private static ProcContext general() {
        ProcContext ctx = new ProcContext();
        ctx.setIndustryKey(Industry.GENERAL);
        return ctx;
    }

    /**
     * 所属行业
     */
    private static ProcContext currentIndustry() {
        ProcContext ctx = general();
        SysTenantDTO tenant = IamUserUtil.currentTenant();
        if (ObjectUtil.isNotNull(tenant)) {
            ctx.setIndustryKey(Industry.fromCode(tenant.getIndustry()));
        }
        return ctx;
    }

    /**
     * 所属客户
     */
    private static ProcContext currentCustomize() {
        ProcContext ctx = new ProcContext();
        ctx.setCustomizeKey(Customize.STANDARD);
        SysTenantDTO tenant = IamUserUtil.currentTenant();
        if (ObjectUtil.isNotNull(tenant)) {
            ctx.setCustomizeKey(Customize.fromCode(tenant.getCustomer()));
        }
        return ctx;
    }
}
