package com.elitesland.cbpl.unicom.register;

import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.unicom.annotation.Unicom;
import com.elitesland.cbpl.unicom.annotation.UnicomReference;
import com.elitesland.cbpl.unicom.config.UnicomProperties;
import com.elitesland.cbpl.unicom.domain.InvokeMode;
import com.elitesland.cbpl.unicom.proxy.UnicomInterfaceProxyFactoryBean;
import com.elitesland.cbpl.unicom.util.ReflectionUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.spring.Constants;
import org.apache.dubbo.config.spring.ReferenceBean;
import org.apache.dubbo.config.spring.reference.ReferenceAttributes;
import org.reflections.Reflections;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;

import javax.annotation.Nonnull;
import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 注册代理类
 *
 * @author eric.hao
 * @since 2022/06/28
 */
@Slf4j
@Configuration
public class ProxyBeanDefinitionRegister implements BeanDefinitionRegistryPostProcessor, EnvironmentAware {

    private UnicomProperties properties;

    @Override
    public void postProcessBeanDefinitionRegistry(@Nonnull BeanDefinitionRegistry beanDefinitionRegistry) throws BeansException {
        Reflections reflections = ReflectionUtil.getReflections(properties);

        Set<Class<?>> typesAnnotatedWith = reflections.getTypesAnnotatedWith(Unicom.class);
        Set<Field> fieldsAnnotatedWith = reflections.getFieldsAnnotatedWith(UnicomReference.class);
        for (Class<?> superClass : typesAnnotatedWith) {
            if (!superClass.isInterface()) {
                continue;
            }
            Unicom anno = superClass.getAnnotation(Unicom.class);
            InvokeMode mode = properties.invokeMode(anno.domain());
            switch (mode) {
                case JVM:
                    jvmRegister(beanDefinitionRegistry, superClass);
                    break;
                case DUBBO:
                    Set<String> versions = fieldsAnnotatedWith.stream()
                            .filter(row -> row.getType().equals(superClass))
                            .map(row -> row.getAnnotation(UnicomReference.class).dv()).collect(Collectors.toSet());
                    versions.forEach(version -> dubboRegister(beanDefinitionRegistry, superClass, version));
                    break;
                case FEIGN:
            }
        }
    }

    private void jvmRegister(BeanDefinitionRegistry beanDefinitionRegistry, Class<?> superClass) {
        RootBeanDefinition beanDefinition = new RootBeanDefinition();
        // 指定 Class 类型
        beanDefinition.setBeanClass(UnicomInterfaceProxyFactoryBean.class);
        // 属性集合
        beanDefinition.getPropertyValues().addPropertyValue("unicomInterface", superClass);
        // 是否是首选的 @Primary
        beanDefinition.setPrimary(true);
        // 装配模式
        beanDefinition.setAutowireMode(RootBeanDefinition.AUTOWIRE_BY_TYPE);
        logger.debug("[UNICOM] jvm register className({}), beanDefinition({}).", superClass.getSimpleName(), beanDefinition);
        beanDefinitionRegistry.registerBeanDefinition(superClass.getSimpleName(), beanDefinition);
    }

    private void dubboRegister(BeanDefinitionRegistry beanDefinitionRegistry, Class<?> superClass, String version) {
        RootBeanDefinition beanDefinition = new RootBeanDefinition();
        // 指定 Class 类型
        beanDefinition.setBeanClass(ReferenceBean.class);
        beanDefinition.setAttribute(ReferenceAttributes.INTERFACE_CLASS, superClass);
        beanDefinition.setAttribute(ReferenceAttributes.INTERFACE_NAME, superClass.getName());
        Map<String, Object> props = new HashMap<>();
        props.put("id", StrUtil.lowerFirst(superClass.getSimpleName()));
        props.put("interface", superClass.getName());
        props.put("version", version);
        beanDefinition.setAttribute(Constants.REFERENCE_PROPS, props);
        // 是否是首选的 @Primary
        beanDefinition.setPrimary(true);
        // 装配模式
        beanDefinition.setAutowireMode(RootBeanDefinition.AUTOWIRE_BY_TYPE);
        logger.debug("[UNICOM] dubbo register className({}), beanDefinition({}).", superClass.getSimpleName(), beanDefinition);
        beanDefinitionRegistry.registerBeanDefinition(superClass.getSimpleName(), beanDefinition);
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory factory) throws BeansException {
    }

    @Override
    public void setEnvironment(Environment environment) {
        this.properties = UnicomProperties.from(environment);
    }
}