package com.elitesland.cbpl.unicom.proxy;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.unicom.adapter.UnicomAdapter;
import com.elitesland.cbpl.unicom.util.ApplicationContextGetBeanHelper;
import com.elitesland.cbpl.unicom.util.ParamAnalyser;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.springframework.cglib.proxy.MethodInterceptor;
import org.springframework.cglib.proxy.MethodProxy;

import java.lang.reflect.Method;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2022/06/28
 */
@Slf4j
public class UnicomInterfaceProxy implements MethodInterceptor {

    private final Reflections reflections;

    public UnicomInterfaceProxy(Reflections reflections) {
        this.reflections = reflections;
    }

    /**
     * 委托方法被调用时转发
     *
     * @param obj         代理子类实例
     * @param method      委托类方法反射
     * @param args        方法参数对象
     * @param methodProxy 用于调用原方法
     */
    @Override
    public Object intercept(Object obj, Method method, Object[] args, MethodProxy methodProxy) throws Throwable {
        List<Object> providers = reflections.getSubTypesOf(method.getDeclaringClass()).stream().map(Class::getName)
                .map(ApplicationContextGetBeanHelper::getBean).collect(Collectors.toList());
        if (CollUtil.isEmpty(providers)) {
            throw new RuntimeException("[UNICOM] " + method.getDeclaringClass() + "：未找到任何实现类");
        }
        Object provider = resolveAdapter(providers);
        return methodProxy.invoke(provider, ParamAnalyser.parse(provider, method, args));
    }

    private Object resolveAdapter(List<Object> providers) {
        UnicomAdapter adapter = loadAdapter();
        // 默认实现类
        if (ObjectUtil.isNull(adapter)) {
            for (Object row : providers) {
                logger.debug("[UNICOM] load impl className({})", row.getClass().getSimpleName());
            }
            Object provider = providers.get(0);
            logger.debug("[UNICOM] execute impl className({})", providers.get(0).getClass().getSimpleName());
            return provider;
        }
        // 自定义规则
        return adapter.filter(providers);
    }

    /**
     * 代理适配实现类
     */
    private UnicomAdapter loadAdapter() {
        List<UnicomAdapter> adapters = reflections.getSubTypesOf(UnicomAdapter.class).stream().map(Class::getName)
                .map(row -> (UnicomAdapter) ApplicationContextGetBeanHelper.getBean(row)).collect(Collectors.toList());
        if (CollUtil.isEmpty(adapters)) {
            return null;
        }
        // 自定义适配规则
        for (UnicomAdapter adapter : adapters) {
            logger.debug("[UNICOM] load adapter className({})", adapter.getClass().getSimpleName());
        }
        UnicomAdapter adapter = adapters.get(0);
        logger.debug("[UNICOM] execute adapter className({})", adapter.getClass().getSimpleName());
        return adapter;
    }
}