package com.elitesland.cbpl.unicom.proxy;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.unicom.adapter.DefaultUnicomAdapter;
import com.elitesland.cbpl.unicom.adapter.UnicomAdapter;
import com.elitesland.cbpl.unicom.annotation.Unicom;
import com.elitesland.cbpl.unicom.config.UnicomProperties;
import com.elitesland.cbpl.unicom.util.ApplicationContextGetBeanHelper;
import com.elitesland.cbpl.unicom.util.ParamAnalyser;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.springframework.cglib.proxy.MethodInterceptor;
import org.springframework.cglib.proxy.MethodProxy;

import java.lang.reflect.Method;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2022/06/28
 */
@Slf4j
public class UnicomInterfaceProxy implements MethodInterceptor {

    private final Reflections reflections;
    private final UnicomProperties properties;

    public UnicomInterfaceProxy(Reflections reflections, UnicomProperties properties) {
        this.reflections = reflections;
        this.properties = properties;
    }

    /**
     * 委托方法被调用时转发
     *
     * @param obj         代理子类实例
     * @param method      委托类方法反射
     * @param args        方法参数对象
     * @param methodProxy 用于调用原方法
     */
    @Override
    public Object intercept(Object obj, Method method, Object[] args, MethodProxy methodProxy) throws Throwable {
        Class<?> superClass = method.getDeclaringClass();
        List<Object> providers = reflections.getSubTypesOf(superClass).stream().map(Class::getName)
                .map(ApplicationContextGetBeanHelper::getBean).collect(Collectors.toList());
        if (CollUtil.isEmpty(providers)) {
            throw new RuntimeException("[UNICOM] " + superClass + "：未找到任何实现类");
        }
        Object provider = loadAdapter(superClass).filter(providers);
        return methodProxy.invoke(provider, ParamAnalyser.parse(provider, method, args));
    }

    /**
     * 代理适配实现类
     */
    private UnicomAdapter loadAdapter(Class<?> superClass) {
        Unicom anno = superClass.getAnnotation(Unicom.class);
        Class<?> adapterClass = anno.adapter();
        if (adapterClass.equals(DefaultUnicomAdapter.class) && StrUtil.isNotEmpty(properties.getDefaultAdapter())) {
            String defaultAdapter = properties.getDefaultAdapter();
            return (UnicomAdapter) ApplicationContextGetBeanHelper.getBean(defaultAdapter);
        }
        return (UnicomAdapter) ApplicationContextGetBeanHelper.getBean(adapterClass.getName());
    }
}