package com.elitesland.cbpl.tool.es.repository;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.tool.es.util.ElasticsearchQuery;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.data.elasticsearch.core.*;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.*;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2023/06/13
 */
@Slf4j
@Component
public class ElasticsearchRepository {

    @Resource
    private ElasticsearchRestTemplate elasticsearchRestTemplate;

    @PostConstruct
    public void init() {
        if (ObjectUtil.isNotNull(elasticsearchRestTemplate)) {
            elasticsearchRestTemplate.setRefreshPolicy(RefreshPolicy.IMMEDIATE);
        }
    }

    /**
     * 判断索引是否存在
     *
     * @param indexName 索引名称
     * @return boolean
     */
    public boolean existsIndex(String indexName) {
        return elasticsearchRestTemplate.indexOps(IndexCoordinates.of(indexName)).exists();
    }

    /**
     * 创建索引
     *
     * @param indexName 索引名称
     * @return boolean
     */
    public boolean createIndex(String indexName) {
        return createIndex(indexName, "", "");
    }

    /**
     * 创建索引
     *
     * @param indexName 索引名称
     * @param policy    索引策略(名称)
     * @param alias     滚动更新别名
     * @return boolean
     */
    public boolean createIndex(String indexName, String policy, String alias) {
        if (existsIndex(indexName)) {
            return true;
        }
        Map<String, Object> setting = new HashMap<>();
        setting.put("index.lifecycle.name", StrUtil.blankToDefault(policy, ""));
        setting.put("index.lifecycle.rollover_alias", StrUtil.blankToDefault(alias, ""));
        boolean result = elasticsearchRestTemplate.indexOps(IndexCoordinates.of(indexName)).create(setting);
        logger.info("[PHOENIX-ES] CREATE INDEX NAME({}) SUCCESS.", indexName);
        return result;
    }

    /**
     * 删除索引
     *
     * @param indexName 索引名称
     * @return boolean
     */
    public boolean deleteIndex(String indexName) {
        if (!existsIndex(indexName)) {
            logger.warn("[PHOENIX-ES] INDEX NAME({}) NOT EXISTED.", indexName);
            return true;
        }
        boolean result = elasticsearchRestTemplate.indexOps(IndexCoordinates.of(indexName)).delete();
        logger.info("[PHOENIX-ES] DELETE INDEX({}) SUCCESS.", indexName);
        return result;
    }

    /**
     * 写入数据 - 单条
     */
    public <T> void insert(T data, String indexName) {
        if (createIndex(indexName)) {
            elasticsearchRestTemplate.save(data, IndexCoordinates.of(indexName));
        }
    }

    /**
     * 写入数据
     */
    public <T> void batchInsert(List<T> dataList, String indexName) {
        if (createIndex(indexName)) {
            List<IndexQuery> indexQueryList = new ArrayList<>();
            dataList.stream().map(ElasticsearchQuery::indexQueryBuilder).forEach(indexQueryList::add);
            elasticsearchRestTemplate.bulkIndex(indexQueryList, IndexCoordinates.of(indexName));
        }
    }

    /**
     * 删除数据 - 单条件
     */
    public Long delete(String indexName, Class<?> clazz, String bizKey, List<String> bizValue) {
        // 匹配文件查询
        BoolQueryBuilder boolQueryBuilder = new BoolQueryBuilder();
        boolQueryBuilder.must(QueryBuilders.termsQuery(bizKey, bizValue));
        NativeSearchQuery query = new NativeSearchQueryBuilder().withQuery(boolQueryBuilder).build();
        // 删除索引数据
        ByQueryResponse resp = elasticsearchRestTemplate.delete(query, clazz, IndexCoordinates.of(indexName));
        logger.info("[PHOENIX-ES] DELETE index({}) bizKey({}) bizValue({}) SUCCESS.", indexName, bizKey, bizValue);
        return resp.getTotal();
    }

    /**
     * 删除数据 - 根据数据ID
     */
    public void deleteById(String id, String indexName) {
        if (ObjectUtil.isNotNull(id) && StringUtils.isNotBlank(indexName)) {
            // 根据索引删除索引id数据
            elasticsearchRestTemplate.delete(id, IndexCoordinates.of(indexName));
            logger.info("[PHOENIX-ES] DELETE index({}) dataId({}) SUCCESS.", indexName, id);
        }
    }

    /**
     * 列表查询
     */
    public <T> List<T> queryList(String indexName, Class<T> clazz, NativeSearchQuery query) {
        SearchHits<T> search = elasticsearchRestTemplate.search(query, clazz, IndexCoordinates.of(indexName));
        if (search.getTotalHits() == 0) {
            return Collections.emptyList();
        }
        List<SearchHit<T>> searchHits = search.getSearchHits();
        return searchHits.stream().map(SearchHit::getContent).collect(Collectors.toList());
    }
}
