package com.elitesland.cloudt.context.spi;

import com.elitesland.cloudt.context.spi.support.ProviderInstanceHolder;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.Optional;

/**
 * SPI工具类入口.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/16
 */
public final class ServiceProviderLoader {

    private ServiceProviderLoader() {
    }

    /**
     * 获取所有的提供者
     *
     * @param serviceType 服务接口
     * @param <T>         服务类型
     * @return 提供者
     */
    public static <T> List<T> loadProviderInstances(@NonNull Class<T> serviceType) {
        return loadProviderInstances(serviceType, null);
    }

    /**
     * 获取所有的提供者
     * <p>
     * 默认首次加载后会放入本地缓存，如果重新加载，则忽略本地缓存
     *
     * @param serviceType 服务接口
     * @param reload      是否重新加载
     * @param <T>         服务类型
     * @return 提供者
     */
    public static <T> List<T> loadProviderInstances(@NonNull Class<T> serviceType, Boolean reload) {
        return ProviderInstanceHolder.loadProviderInstances(serviceType, reload);
    }

    /**
     * 获取一个提供者
     * <p>
     * 如果有默认的，优先返回默认的
     *
     * @param serviceType 服务接口
     * @param <T>         服务类型
     * @return 提供者
     */
    public static <T> Optional<T> loadProviderInstanceOne(@NonNull Class<T> serviceType) {
        return loadProviderInstanceOne(serviceType, null);
    }

    /**
     * 获取一个提供者
     * <p>
     * 默认首次加载后会放入本地缓存，如果重新加载，则忽略本地缓存
     * 如果有默认的，优先返回默认的
     *
     * @param serviceType 服务接口
     * @param reload      是否重新加载
     * @param <T>         服务类型
     * @return 提供者
     */
    public static <T> Optional<T> loadProviderInstanceOne(@NonNull Class<T> serviceType, Boolean reload) {
        return ProviderInstanceHolder.loadProviderInstance(serviceType, reload);
    }

}
