package com.elitesland.cloudt.context;

import com.elitesland.cloudt.context.util.DatetimeUtil;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.zaxxer.hikari.HikariConfig;
import lombok.extern.log4j.Log4j2;
import org.apache.shardingsphere.shardingjdbc.jdbc.core.connection.ShardingConnection;
import org.springframework.aop.interceptor.AsyncUncaughtExceptionHandler;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.task.TaskExecutor;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.AsyncConfigurer;
import org.springframework.scheduling.annotation.EnableAsync;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.concurrent.Executor;

/**
 * Cloudt Context主配置类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/18
 */
@Configuration(proxyBeanMethods = false)
@Import({CloudtContextAutoConfiguration.JacksonConfiguration.class, CloudtContextAutoConfiguration.AsyncConfiguration.class,
        CloudtContextAutoConfiguration.CloudtShardingDataSourceConfiguration.class})
@Order(Ordered.HIGHEST_PRECEDENCE)
@Log4j2
public class CloudtContextAutoConfiguration implements ApplicationContextAware {

    @Override
    public void setApplicationContext(@NonNull ApplicationContext applicationContext) throws BeansException {
        onApplicationContext(applicationContext);
    }

    private void onApplicationContext(ApplicationContext applicationContext) {
        SpringContextHolder.setApplicationContext(applicationContext);
    }

    @Configuration(proxyBeanMethods = false)
    static class JacksonConfiguration {

        @Bean
        public Jackson2ObjectMapperBuilderCustomizer jackson2ObjectMapperBuilderCustomizer() {
            return jacksonObjectMapperBuilder -> jacksonObjectMapperBuilder.failOnUnknownProperties(false)
                    .failOnEmptyBeans(false)
                    .serializerByType(Long.class, ToStringSerializer.instance)
                    .serializerByType(Long.TYPE, ToStringSerializer.instance)
                    .serializerByType(LocalDateTime.class, new LocalDateTimeSerializer(DatetimeUtil.FORMATTER_DATETIME))
                    .deserializerByType(LocalDateTime.class, new LocalDateTimeDeserializer(DatetimeUtil.FORMATTER_DATETIME))
                    .serializerByType(LocalDate.class, new LocalDateSerializer(DatetimeUtil.FORMATTER_DATE))
                    .deserializerByType(LocalDate.class, new LocalDateDeserializer(DatetimeUtil.FORMATTER_DATE));
        }

        @Bean
        public LocalDateTimeSerializer localDateTimeSerializer() {
            return new LocalDateTimeSerializer(DatetimeUtil.FORMATTER_DATETIME);
        }

        @Bean
        public LocalDateTimeDeserializer localDateTimeDeserializer() {
            return new LocalDateTimeDeserializer(DatetimeUtil.FORMATTER_DATETIME);
        }
    }

    @Configuration(proxyBeanMethods = false)
    @ConditionalOnClass(name = "org.apache.shardingsphere.shardingjdbc.jdbc.core.datasource.ShardingDataSource")
    static class CloudtShardingDataSourceConfiguration {

        private final DataSource defaultDataSource;

        public CloudtShardingDataSourceConfiguration(DataSource dataSource) {
            this.defaultDataSource = dataSource;
        }

        @Bean
        @ConditionalOnMissingBean
        public HikariConfig cloudtFlywayHikariConfig() {
            ShardingConnection connection = null;
            try (Connection conn = defaultDataSource.getConnection()) {
                if (conn instanceof ShardingConnection) {
                    connection = (ShardingConnection) conn;
                }
            } catch (SQLException e) {
                log.warn("获取数据库连接失败", e);
                return null;
            }

            if (connection == null) {
                log.warn("使用了shardingjdbc，未获取到有效的连接");
                return null;
            }
            for (Map.Entry<String, DataSource> entry : connection.getDataSourceMap().entrySet()) {
                if (entry.getValue() instanceof HikariConfig) {
                    return (HikariConfig) entry.getValue();
                }
            }

            return null;
        }
    }

    @Configuration(proxyBeanMethods = false)
    @EnableAsync
    @Log4j2
    static class AsyncConfiguration implements AsyncConfigurer {

        private final TaskExecutor taskExecutor;

        public AsyncConfiguration(TaskExecutor taskExecutor) {
            this.taskExecutor = taskExecutor;
        }

        @Override
        public Executor getAsyncExecutor() {
            return taskExecutor;
        }

        @Override
        public AsyncUncaughtExceptionHandler getAsyncUncaughtExceptionHandler() {
            return (ex, method, params) -> log.error("异步线程【" + method.getDeclaringClass().getName() + "." + method.getName() + "】执行异常：", ex);
        }
    }
}
