package com.elitesland.cloudt.context.flyway;

import cn.hutool.core.lang.Assert;
import com.elitesland.cloudt.context.flyway.support.FlywayHelper;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.log4j.Log4j2;
import org.flywaydb.core.Flyway;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.jdbc.JdbcTemplateAutoConfiguration;
import org.springframework.boot.autoconfigure.orm.jpa.HibernateJpaAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;

/**
 * flyway相关自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/23
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnProperty(prefix = FlywayCloudtProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(FlywayCloudtProperties.class)
@ConditionalOnClass(Flyway.class)
@AutoConfigureAfter({DataSourceAutoConfiguration.class, JdbcTemplateAutoConfiguration.class,
        HibernateJpaAutoConfiguration.class})
@Import(FlywayCloudtAutoConfiguration.FlywayAutoMigrate.class)
@Log4j2
public class FlywayCloudtAutoConfiguration {

    private final FlywayCloudtProperties properties;
    private final DataSource defaultDataSource;

    public FlywayCloudtAutoConfiguration(FlywayCloudtProperties properties, DataSource defaultDataSource) {
        this.properties = properties;
        this.defaultDataSource = defaultDataSource;
    }

    @Bean
    @ConditionalOnMissingBean
    public FlywayBuilder flywayBuilder(ObjectProvider<HikariConfig> hikariConfigObjectProvider) {

        var dataSource = buildDataSource(hikariConfigObjectProvider.getIfAvailable());
        return new FlywayBuilder(dataSource, properties);
    }

    @Bean
    public FlywayHelper flywayHolder() {
        return new FlywayHelper(properties);
    }

    private DataSource buildDataSource(HikariConfig defaultHikariConfig) {
        if (StringUtils.hasText(properties.getDbUrl())) {
            HikariConfig config = new HikariConfig();
            config.setJdbcUrl(properties.getDbUrl());
            config.setDriverClassName(Assert.notBlank(properties.getDbDriverClassName(), "dbDriverClassName为空"));
            config.setUsername(Assert.notBlank(properties.getDbUser(), "dbUser为空"));
            config.setPassword(Assert.notBlank(properties.getDbPassword(), "dbPassword为空"));
            return new HikariDataSource(config);
        }

        if (defaultHikariConfig != null) {
            return defaultHikariConfig instanceof DataSource ? (DataSource) defaultHikariConfig : new HikariDataSource(defaultHikariConfig);
        }

        return defaultDataSource;
    }

    static class FlywayAutoMigrate implements ApplicationRunner {

        @Autowired
        private FlywayCloudtProperties flywayCloudtProperties;
        @Autowired
        private FlywayBuilder flywayBuilder;
        @Autowired
        private FlywayHelper flywayHelper;
        @Autowired
        private TaskExecutor taskExecutor;

        @Override
        public void run(ApplicationArguments args) throws Exception {
            taskExecutor.execute(() -> {
                log.info("检查【系统】数据库脚本...");
                checkSysScript();
                log.info("检查【系统】数据库脚本结束");
            });
        }

        private void checkSysScript() {
            if (flywayCloudtProperties.getAutoMigrateAfterStart().isNegative() || flywayCloudtProperties.getAutoMigrateAfterStart().isZero()) {
                log.info("检查【系统】数据库脚本忽略");
                return;
            }

            flywayHelper.migrate(flywayBuilder::createFlywayForSys);
        }
    }
}
